/*
 * Copyright (c) 2023, Azul Systems, Inc. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

// Mirror: none

package java.text;

import java.text.StringCharacterIterator;
import sun.text.CollatorUtilities;
import jdk.internal.icu.text.NormalizerBase;

final class SCINormalizedAzulIterator implements NormalizedAzulIterator {
    private StringCharacterIterator textIterator;
    private NormalizerBase.Mode mode;
    private RuleBasedCollator owner;
    private int count;

    private SCINormalizedAzulIterator(StringCharacterIterator textIterator, NormalizerBase.Mode mode, RuleBasedCollator owner, int count) {
        this.textIterator = textIterator;
        this.mode = mode;
        this.owner = owner;
        this.count = count;
    }

    public SCINormalizedAzulIterator(String sourceText, RuleBasedCollator owner) {
        mode = CollatorUtilities.toNormalizerMode(owner.getDecomposition());
        textIterator = new StringCharacterIterator(NormalizerBase.normalize(sourceText, mode, 0));
        this.owner = owner;
    }

    @Override
    public int next() {
        char c1 = textIterator.current();
        int result = c1;
        textIterator.next();
        if (Character.isHighSurrogate(c1)) {
            char c2 = textIterator.current();
            if (Character.isLowSurrogate(c2)) {
                textIterator.next();
                result = Character.toCodePoint(c1, c2);
            }
        }
        count++;
        return result;
    }

    @Override
    public int previous() {
        char c2 = textIterator.previous();
        int result = c2;
        if (Character.isLowSurrogate(c2)) {
            char c1 = textIterator.previous();
            if (Character.isHighSurrogate(c1)) {
                result = Character.toCodePoint(c1, c2);
            } else {
                textIterator.next();
            }
        }
        count--;
        return result;
    }

    @Override
    public void reset() {
        textIterator.setIndex(textIterator.getBeginIndex());
    }

    @Override
    public boolean updateMode() {
        NormalizerBase.Mode ownerMode = CollatorUtilities.toNormalizerMode(owner.getDecomposition());
        return mode == ownerMode;
    }

    @Override
    public int done() {
        return StringCharacterIterator.DONE;
    }

    @Override
    public Object clone() {
        return new SCINormalizedAzulIterator((StringCharacterIterator) textIterator.clone(), mode, owner, count);
    }

    public int getCount() {
        return count;
    }
}
