//===-- QueryCacheImpl.h ---------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// Implementation file for QueryCache.h.  Don't include this directly!
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_SUPPORT_AZUL_QUERY_CACHE_SERIALIZATION_IMPL_H
#define LLVM_SUPPORT_AZUL_QUERY_CACHE_SERIALIZATION_IMPL_H

#include "QueryCache.h"
#include "QueryCacheSerialization.h"

namespace azul {

namespace qc_detail {
constexpr const char *kOptionalTag_None = "None";
constexpr const char *kOptionalTag_Some = "Some";

/// A StringSwitch-like thing created from \c TextualTraits that matches a
/// string to its corresponding query type.  The entry point is the \c run
/// method which returns -1 on failure or the matching query kind.  I'd have
/// preferred to use llvm::StringSwitch here, but that works only on constant
/// char *s.
/// @{
template <int Idx, template <int> class TextualTraits>
struct QCTraitsAsStringSwitch {
  LLVM_ATTRIBUTE_ALWAYS_INLINE static int run(llvm::StringRef Name) {
    if (Name == TextualTraits<Idx - 1>::queryKindString())
      return Idx - 1;
    typedef QCTraitsAsStringSwitch<Idx - 1, TextualTraits> NextTy;
    return NextTy::run(Name);
  }
};

template <template <int> class TextualTraits>
struct QCTraitsAsStringSwitch<0, TextualTraits> {
  LLVM_ATTRIBUTE_ALWAYS_INLINE static int run(llvm::StringRef) { return -1; }
};
/// @}

/// Helper to lex out a result prefix, like "result None" or "result Sone".
/// @{
enum class LexedResultType {
  SOME,
  NONE,

  MAX
};

LexedResultType lexResultPrefix(qc_utils::QCLexer &QCP);
/// @}

/// Serialize a query \p Q into \p OS.  \p QueryKindName is the name of the
/// query kind (for convenience, this can be computed in \c serializeQuery too).
void serializeQuery(const Query *Q, llvm::StringRef QueryKindName,
                    llvm::raw_ostream &OS);
}

template <typename QCTy, template <int> class TextualTraits>
template <typename... BatonTys>
void QueryCacheSerializer<QCTy, TextualTraits>::serialize(
    QCTy &QC, llvm::raw_ostream &OS, BatonTys &... Batons) {
  OS << "Azul Query Cache\n";
  OS << "Version 0\n";

  typedef QueryCacheSerializer<QCTy, TextualTraits> SelfTy;
  for (Query *Q : make_range(QC.begin(), QC.end()))
    // In order to serialize a query of the kind KindTy we need to use proper
    // compile time instantiation of TextualTraits and QueryCache methods.
    // This ForEachMapKind is essentially a switch by a runtime Q->getKind()
    // value to the proper serialization instantiation.
    qc_detail::ForEachMapKind<SelfTy, QCTy::KindsMax>::run(QC, Q, OS, Batons...);
}

template <typename QCTy, template <int> class TextualTraits>
template <int Idx, typename... BatonTys>
qc_detail::ForEachDirective
QueryCacheSerializer<QCTy, TextualTraits>::forEachMapKind(
    QCTy &QC, Query *Q, llvm::raw_ostream &OS, BatonTys &... Batons) {
  if (Q->getKind() != Idx)
    return qc_detail::ForEachDirective::Continue;

  qc_detail::serializeQuery(Q, TextualTraits<Idx>::queryKindString(), OS);
  OS << " ";
  auto Result = QC.template lookupByQuery<Idx>(Q);
  serializeResult<Idx>(Result, OS, Batons...);
  OS << "\n";
  
  return qc_detail::ForEachDirective::Stop;
}

template <typename QCTy, template <int> class TextualTraits>
template <int Idx, typename... BatonTys>
void QueryCacheSerializer<QCTy, TextualTraits>::serializeResult(
    const std::optional<typename TextualTraits<Idx>::ResultTy> &Result,
    llvm::raw_ostream &OS, BatonTys... Batons) {
  OS << "result ";
  if (!Result.has_value()) {
    OS << qc_detail::kOptionalTag_None;
    return;
  }

  OS << qc_detail::kOptionalTag_Some << " ";
  TextualTraits<Idx>::serializeResult(Result.value(), OS, Batons...);
}

template <typename T>
typename std::enable_if<std::is_integral<T>::value, T>::type
qc_utils::QCLexer::nextTokenAsInt(const char *What) {
  llvm::StringRef Tok = nextToken(What);
  if (hasFailed())
    return T(0);

  T Result = 0;
  if (Tok.getAsInteger<T>(10, Result)) {
    setFailed("wanted integer for `%s', got `%s'!", What, Tok.str().c_str());
    return 0;
  }

  return Result;
}

template <typename QCTy, template <int> class TextualTraits>
bool QueryCacheDeserializer<QCTy, TextualTraits>::stringToQueryKind(
    llvm::StringRef Name, int &QueryKind) {
  typedef qc_detail::QCTraitsAsStringSwitch<QCTy::KindsMax, TextualTraits>
      SwitchTy;
  QueryKind = SwitchTy::run(Name);
  return QueryKind == -1;
}

template <typename QCTy, template <int> class TextualTraits>
template <typename... BatonTys>
bool QueryCacheDeserializer<QCTy, TextualTraits>::deserialize(
    QCTy &QC, qc_utils::QCLexer &QCP,
    std::function<bool(llvm::StringRef)> SkipSection, BatonTys &... Batons) {
  QCP.consumeExactTokenSeq("header", {"Azul", "Query", "Cache"});
  if (QCP.hasFailed())
    return true;

  QCP.consumeExactTokenSeq("version", {"Version", "0"});
  if (QCP.hasFailed())
    return true;

  while (!QCP.isEOF()) {
    auto Next = QCP.nextToken("cache or query or comment");
    if (QCP.hasFailed())
      return true;
    if (Next.equals("cache")) {
      QCP.advanceToNextLine("ignored cache line");
      continue;
    }
    
    if (Next.equals("#") || Next.equals("//")) {
      // Warning: comments *require* something in a seperate token on the same
      // line.  Otherwise, the following line is skipped.
      QCP.advanceToNextLine("comment line");
      continue;
    }
    
    if (!Next.equals("query")) {
      QCP.setFailed("expected a query line!");
      return true;
    }
    
    unsigned NumArgs = QCP.nextTokenAsInt<unsigned>("# arguments");
    if (QCP.hasFailed())
      return true;
    
    llvm::StringRef QueryKindStr = QCP.nextToken("query kind");
    if (QCP.hasFailed())
      return true;
    
    if (SkipSection && SkipSection(QueryKindStr)) {
      // Bypass empty records for tables we no longer support, but that might be
      // sitting around in stale files.
      QCP.advanceToNextLine("dead line");
      if (QCP.hasFailed())
        return true;
      continue;
    }

    int QueryKind = 0;
    if (stringToQueryKind(QueryKindStr, QueryKind)) {
      QCP.setFailed("could not recognize query type!");
      return true;
    }

    assert(QueryKind < QCTy::KindsMax &&
           "stringToQueryKind should have failed otherwise!");

    typedef QueryCacheDeserializer<QCTy, TextualTraits> SelfTy;
    typedef qc_detail::ForEachMapKind<SelfTy, QCTy::KindsMax> ForEachTy;
    ForEachTy::run(QC, QCP, QueryKind, NumArgs, Batons...);
    if (QCP.hasFailed())
      return true;
  }

  return false;
}

template <typename QCTy, template <int> class TextualTraits>
template <int QueryKind, typename... BatonTys>
qc_detail::ForEachDirective
QueryCacheDeserializer<QCTy, TextualTraits>::forEachMapKind(
    QCTy &QC, qc_utils::QCLexer &QCP, const int ActualQueryKind,
    unsigned NumArgs, BatonTys &... Batons) {
  if (ActualQueryKind != QueryKind)
    return qc_detail::ForEachDirective::Continue;

  Query *Q = qc_detail::deserializeQuery(QCP, QC, QueryKind, NumArgs);
  if (QCP.hasFailed())
    return qc_detail::ForEachDirective::Stop;

  assert(Q->getKind() == QueryKind && "Should have failed above otherwise!");
  
  if (QC.template lookupByQuery<QueryKind>(Q)) {
    QCP.setFailed("Repeated query!");
    return qc_detail::ForEachDirective::Stop;
  }

  auto ResultPrefix = qc_detail::lexResultPrefix(QCP);
  if (QCP.hasFailed())
    return qc_detail::ForEachDirective::Stop;

  if (ResultPrefix == qc_detail::LexedResultType::NONE) {
    QC.template insert<QueryKind>(Q, std::nullopt);
    return qc_detail::ForEachDirective::Stop;
  }

  assert(ResultPrefix == qc_detail::LexedResultType::SOME &&
         "lexResultPrefix should have failed the parse otherwise!");

  auto MaybeResult =
      TextualTraits<QueryKind>::deserializeResult(QCP, Batons...);
  if (QCP.hasFailed())
    return qc_detail::ForEachDirective::Stop;

  QC.template insert<QueryKind>(Q, *MaybeResult);
  return qc_detail::ForEachDirective::Stop;
}

#ifndef NDEBUG
template <int QueryKindsMax, template <int> class QueryTraits>
template <int Idx, typename ActionTy>
qc_detail::ForEachDirective
QueryCache<QueryKindsMax, QueryTraits>::forEachMapKind(ActionTy Action) {
  for (const auto &Val : getMap<Idx>()) {
    assert(Val.first->getKind() == Idx && "Query in wrong table!");
    Action(Idx, Val.first);
  }
  return qc_detail::ForEachDirective::Continue;
}

template <int QueryKindsMax, template <int> class QueryTraits>
template <typename ActionTy>
void QueryCache<QueryKindsMax, QueryTraits>::verify(ActionTy A) {
  typedef QueryCache<QueryKindsMax, QueryTraits> SelfTy;
  qc_detail::ForEachMapKind<SelfTy, QueryKindsMax>::run(*this, A);
}
#endif

} // end namespace azul

#endif
