//===-- VMInterface.h - Interface to the virtual machine --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header the query API Orca should use to query facts from the VM.  This
// interface hides all the details about logging and replay from the rest of
// Orca.
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_VMINTERFACE_H
#define LLVM_AZUL_VMINTERFACE_H

#include "llvm/Analysis/Orca/TypeUtils.h"
#include "llvm/ADT/StringRef.h"
#include "llvm/IR/Function.h"

#include <optional>

namespace azul {
struct ValueInfo;

namespace orca {
class JavaTypeInfo;
struct FieldInfo;
class FunctionInfo;

/// Codification of which integer stands for which generation.
///
/// The generation of an object for generational GC's Orca understands are not
/// allowed to change unless the object is piped through a gc_relocate.  Piping
/// an object through a gc_relocate can promote an object in the new generation
/// to the old generation, cannot change the generation of objects in the old
/// generation.
///
/// NB! We still represent object generations as integers "over the wire"
/// including serialization via the query-cache.  This is enum is just a
/// convenient way to not mispsell the integers.
enum ObjectGeneration {
  OG_NewGeneration = 1,
  OG_OldGeneration = 2,
  OG_Invalid = 3
};
}

namespace VMInterface {

std::optional<bool> isSubtypeOf(llvm::LLVMContext &C,
                                TypeUtils::CompileTimeKlassID parent,
                                TypeUtils::CompileTimeKlassID child);

std::optional<bool> requiresFinalizer(llvm::LLVMContext &C,
                                      const TypeUtils::JavaType &T);

// The callerName can be an empty stringRef if we are only interested in the
// callee being returned.
std::optional<llvm::Function *> getFunctionBody(llvm::Function *F);

std::optional<llvm::Function *>
getMergedAllocFunc(llvm::LLVMContext &C, llvm::Module *M,
                   llvm::ArrayRef<TypeUtils::CompileTimeKlassID> KlassIDs);

void inliningSuccessful(llvm::LLVMContext &C, llvm::Function *caller,
                        llvm::Function *callee, bool isJavaMethod);

/// Notifies the VM about inlining decision for a Java method call site.
/// Callee == std::nullopt indicates an indirect call site. Such a call site
/// can not be inlined. I.e. Success must be false if Callee is std::nullopt.
void reportJavaMethodInliningDecision(llvm::LLVMContext &C, 
                    llvm::Function *Caller,
                    std::optional<llvm::Function *> Callee,
                    llvm::ArrayRef<uint64_t> CallerIDs,
                    llvm::ArrayRef<uint64_t> CallerBCIs,
                    llvm::StringRef Message, bool Success);

/// Queries the VM advice about inlining decision on a particular call site
/// described by { \p Caller, \p Callee, \p CallerIDs, \p CallerBCIs }.
/// If the VM has no preference regarding this call site, returns std::nullopt.
/// If the VM says to inline or not inline this call site,
/// returns Some(true) or Some(false) respectively.
std::optional<bool> getVMInliningAdvice(llvm::LLVMContext &C,
                                        const llvm::Function *Caller,
                                        const llvm::Function *Callee,
                                        llvm::ArrayRef<uint64_t> CallerIDs,
                                        llvm::ArrayRef<uint64_t> CallerBCIs);

/// If the offset into the given klass (specified via an opaque Klass ID ) is
/// known to be constant, return the value.  Note that the high bits of the
/// returned value must be ignored if the size of the load is less than 8
/// bytes.  It is not specified as to whether the value is sign or zero
/// extended.
std::optional<uint64_t> getLoadResultKlass(llvm::LLVMContext &C,
                                           TypeUtils::CompileTimeKlassID Kid,
                                           int64_t Offset, uint32_t Size);

/// If the offset into the given value (specified via a known value ID) is
/// known to be constant, return the value.  Note that the high bits of the
/// returned value must be ignored if the size of the load is less than 8
/// bytes.  It is not specified as to whether the value is sign or zero
/// extended.
std::optional<uint64_t> getLoadResultKnownValue(llvm::LLVMContext &C,
                                                uint64_t KnownValueID,
                                                int64_t Offset, uint32_t Size);

/// Given a constant address we're loading from, try to find a known value id
/// which can be used for that load.  This has to be done lazily (rather than
/// just in the frontend) to support constant folding.
std::optional<uint64_t> getKnownValueKey(llvm::LLVMContext &C,
                                         uint64_t ConstAddr);

/// Given a constant offset into a known object, check to see if the result
/// itself is a known object.  Note that this callback only applies to loads of
/// pointer types; it is complementry but non-overlaping with the
/// getLoadResultKnownValue routine above.
std::optional<uint64_t> getKnownValueKeyDependent(llvm::LLVMContext &C,
                                                  uint64_t KnownValueID,
                                                  int64_t Offset);

/// Given an Offset and Size of a field within an object kid, check to see if the
/// field is invariant with respect to the callee calleeName.
std::optional<bool> isUnmodifiedByCall(llvm::LLVMContext &C,
                                       TypeUtils::CompileTimeKlassID kid,
                                       int64_t Offset, uint32_t Size,
                                       llvm::StringRef calleeName);

/// Given an Offset and Size of a field within an object kid, check to see if
/// the field is invariant with respect to the indirect callee.
std::optional<bool> isUnmodifiedByJavaCall(
    llvm::LLVMContext &C, TypeUtils::CompileTimeKlassID FieldHolderKid,
    int64_t Offset, uint32_t Size, uint64_t calledID, uint64_t callerBCI);

/// If the class/interface specified by the kid argument has unique concrete
/// subtype returns klass id of this subtype.
/// If needLeafType argument is true the subtype is not allowed to have
/// subclasses.
std::optional<TypeUtils::CompileTimeKlassID>
getUniqueSubtype(llvm::LLVMContext &C, TypeUtils::CompileTimeKlassID kid,
                 bool needLeafType);

/// Get kid for the elements in the array. ArrayKid must be object array.
std::optional<TypeUtils::CompileTimeKlassID>
getObjArrayElementKid(llvm::LLVMContext &C,
                      TypeUtils::CompileTimeKlassID ArrayKid);

std::optional<llvm::Function *> getInlineCacheCallStub(llvm::LLVMContext &C,
                                                       llvm::Module *M,
                                                       uint64_t CallerID,
                                                       uint64_t CallerBCI);

std::optional<uint64_t> getVMIntegerConstant(llvm::LLVMContext &C,
                                             llvm::StringRef ConstantName);

/// Return information about an object implied by it's type (and if it's an
/// array and the length is known, it's length).
std::optional<azul::orca::JavaTypeInfo>
getJavaTypeInfo(llvm::LLVMContext &C, const TypeUtils::JavaType &T,
                std::optional<uint64_t> ArrayLen);

/// Returns FieldInfo description for the given offset of the given type. A
/// successful query doesn't guarantee that the location being asked about is
/// dereferenceable. For example, if T describes an array type, the returned
/// FieldInfo would describe an array element assuming that it's in bounds. It's 
/// up to the caller to check dereferenceability if this property is required.
std::optional<azul::orca::FieldInfo> 
getFieldInfoAtOffset(llvm::LLVMContext &C, TypeUtils::JavaType T, bool IsNew, 
                     int64_t Offset);

std::optional<TypeUtils::CompileTimeKlassID>
leastCommonAncestor(llvm::LLVMContext &C,
                    TypeUtils::CompileTimeKlassID KlassID1,
                    TypeUtils::CompileTimeKlassID KlassID2);

/// For the given known-value object returns its hash code
std::optional<uint64_t> getObjectHashcode(llvm::LLVMContext &C, uint64_t KVID);

/// For the given known-value object returns if its generation is known.  Return
/// \c OG_NewGeneration for the new generation, and \c OG_OldGeneration for the
/// old generation.
std::optional<azul::orca::ObjectGeneration>
getObjectGeneration(llvm::LLVMContext &C, uint64_t KVID);

/// For the given known-value object returns its klass ID
std::optional<azul::TypeUtils::JavaType> 
getObjectJavaType(llvm::LLVMContext &C, uint64_t KVID);

/// For the given known-value object of java.lang.Class type
/// returns the klass ID of the corresponding java object
std::optional<azul::TypeUtils::CompileTimeKlassID>
getJavaTypeForJavaLangClass(llvm::LLVMContext &C, uint64_t KVID);

/// Returns the specialized implementation for a call site marked with
/// "specializable-function" attribute.
///
/// \param FunctionName The name of the function we want to specialize.
/// \param ArgumentInfo The arguments facts to specialize for.
/// \param CallerIDs The caller IDs from the JVM state or empty if there is no
/// JVM state.
/// \param CallerBCIs The caller BCIs from the JVM state or empty if there is no
/// JVM state.
std::optional<llvm::Function *> getSpecializedImplementation(
    llvm::LLVMContext &C, llvm::Module *M,
    llvm::StringRef FunctionName,
    llvm::ArrayRef<azul::ValueInfo> ArgumentInfo,
    llvm::StringRef CallerName,
    llvm::ArrayRef<uint64_t> CallerIDs,
    llvm::ArrayRef<uint64_t> CallerBCIs);

/// Returns the constant result for a call site marked with "foldable-function"
/// attribute.
///
/// \param FunctionName The name of the called function.
/// \param ArgumentInfo The arguments facts to constant fold for.
/// \param CallerIDs The caller IDs from the JVM state or empty if there is no
/// JVM state.
/// \param CallerBCIs The caller BCIs from the JVM state or empty if there is no
/// JVM state.
std::optional<uint64_t>
getConstantResult(llvm::LLVMContext &C, llvm::StringRef FunctionName,
                  llvm::ArrayRef<azul::ValueInfo> ArgumentInfo,
                  llvm::ArrayRef<uint64_t> CallerIDs,
                  llvm::ArrayRef<uint64_t> CallerBCIs);


/// If the given known-value pointer can be loaded from a global variable 
/// returns the name of this global.
std::optional<std::string>
getKnownValueGlobalName(llvm::LLVMContext &C, uint64_t KVID);
/// If the given global represents a known value returns its ID
std::optional<uint64_t>
getKnownValueID(llvm::LLVMContext &C, llvm::StringRef GlobalName);

/// For the given function ID returns the FunctionInfo describing this function.
std::optional<azul::orca::FunctionInfo>
getFunctionInfo(llvm::LLVMContext &C, uint64_t FunctionID);

/// Notifies the VM about function specialization and requests a new ID for
/// the specialized copy.
///
/// If the request succeeds the VM returns a new ID. The specialized copy should
/// be patched to update the function ID using azul::Utils::ReplaceFunctionID.
/// If the request fails std::nullopt is returned and the specialization can not be
/// done.
std::optional<uint32_t>
getSpecializedFunctionID(llvm::LLVMContext &C, llvm::StringRef OriginalName,
                         llvm::StringRef SpecializedName);

/// Returns true if the given known value object is a cached box object.
/// In other words it returns true iff
///   BoxType.valueOf(kvid.boxTypeValue) == kvid.
std::optional<bool>
isCachedBoxObject(llvm::LLVMContext &C, uint64_t KnownValueID);

std::optional<uint64_t> 
getBoxedValueOffset(llvm::LLVMContext &C, TypeUtils::JavaType T);

std::optional<azul::TypeUtils::CompileTimeKlassID>
runTimeToCompileTimeKlassID(llvm::LLVMContext &C, uint64_t RTKID);

std::optional<uint64_t>
compileTimeToRunTimeKlassID(llvm::LLVMContext &C,
                            azul::TypeUtils::CompileTimeKlassID CTKID);
}
}

#endif
