//===-- Utils.h -------------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef LLVM_UTILS_H
#define LLVM_UTILS_H

#include "llvm/ADT/APInt.h"
#include "llvm/Analysis/Orca/TypeUtils.h"
#include "llvm/Analysis/BlockFrequencyInfo.h"
#include "llvm/IR/BasicBlock.h"
#include "llvm/IR/IRBuilder.h"
#include "llvm/IR/DataLayout.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/Metadata.h"
#include "llvm/IR/Module.h"
#include "llvm/IR/Value.h"
#include "llvm/Support/Allocator.h"
#include "llvm/Support/Casting.h"
#include "llvm/Support/CommandLine.h"
#include <functional>
#include <memory>
#include <optional>
#include <utility>

namespace llvm {
class DominatorTree;
class Function;
class InlineCost;
class InlineFunctionInfo;
class Instruction;
class LLVMContext;
class LoadInst;
class Loop;
class StoreInst;
class ScalarEvolution;
class TargetLibraryInfo;
class formatted_raw_ostream;

bool isInliningCandidate(const llvm::Function &F);
bool isInliningCandidate(const llvm::GlobalValue &F);

bool isGeneratable(const llvm::Function &F);
bool isGeneratable(const llvm::GlobalValue &F);

typedef uint32_t BranchWeightType;
}

namespace azul {
namespace Utils {

/// Match an address of the form Base + DispC + i * Scale.
bool matchAddress(const llvm::DataLayout &DL, llvm::Value *Addr,
                  llvm::Value *&Base, uint64_t &DispCon, uint64_t &ScaleCon,
                  llvm::Value *&UnknownIndex);

// Returns true if the module has explicit relocations.
bool hasExplicitRelocations(const llvm::Module &M);

// This returns true when we can have multiple functions living in the module.
bool relaxAssertionsForMultiFunctionModule();

// This is valid only after explicit relocations are done, since all pollable
// calls are wrapped in a statepoint.
void getGCPollableBBsInFunc(llvm::Function &F,
                      llvm::DenseSet<llvm::BasicBlock *> &GCPollableBBs);

// Identify deoptimize block before and after explicit relocations
// (rewriteStatepointsForGC). This is different from
// BasicBlock::getTerminatingDeoptimizeCall, which only works before explicit
// relocations.
// We intentionally keep them separate because explicit relocations is not a
// mandatory functional pass upstream. In particular, we have usage of
// `getTerminatingDeoptimizeCall` in SelectionDAG which requires identifying the
// "non relocated version" of deoptimize intrinsic.
// TODO: This can be upstreamed by adding a default parameter in
// getTerminatingDeoptimizeCall to check for deoptimize wrapped in statepoints
// as well.
bool isDeoptimizeBlock(const llvm::BasicBlock &BB);

// Returns true if the module has compressed oops.
bool hasCompressedOops(const llvm::Module *M);

// Returns true if function has allocation instruction.
bool hasAllocations(const llvm::Function &F);

// Returns true if V is a compressed oop.
bool isCompressedOop(const llvm::Value *V);

// Returns true if T is compressed oop type (either scalar or vector type).
bool isCompressedOopType(const llvm::Type *T);

// Retuns true if V is the compress call (scalar or vector version).
bool isCompressCall(const llvm::Value &V);

// Retuns true if V is the uncompress call (scalar or vector version).
bool isUncompressCall(const llvm::Value &V);

// Returns true if V is the compress or uncompress call (scalar or vector
// version).
bool isCompressOrUncompress(const llvm::Value &V);

// Returns true if we need to optimize for compressed pointers.
bool OptimizeForCompressedPointers();

// Compresses the Arg by calling azul.compress.
llvm::Value *CreateCompressedOop(llvm::Value *Arg, llvm::IRBuilder<> &B);

// Uncompress the Arg by calling azul.uncompress.
llvm::Value *CreateUncompressedOop(llvm::Value *Arg, llvm::IRBuilder<> &B);

// This function returns the underlying pointer (either the compressed or
// uncompressed type) depending on what Call does.
// If call is neither compress or uncompress, then return null.
llvm::Value *
getUnderlyingPointerFromCompressUncompress(const llvm::Value *Call);

// Type utilities for conversion to and from compressed type (handles scalar and
// vector source types).
llvm::Type *ConvertToUncompressedType(const llvm::Type *CompSrcType,
                                      llvm::LLVMContext &C);
llvm::Type *ConvertToCompressedType(const llvm::Type *UncompSrcType,
                                    llvm::LLVMContext &C);

// Utilities for generating a compressed load and store.

// Creates a typed load - either primitive or pointer load.
// For a primitive type loaded value, this is just a regular load and both
// values of the returned pair is the load itself.
// For a pointer type loaded value, the first element in the pair is the load
// itself. In compressed-oop mode, the second element is the uncompressed
// pointer value.  In regular mode, the second element will is the load itself.
std::pair<llvm::LoadInst *, llvm::Value *>
CreateTypedLoad(llvm::Value *Addr, llvm::Type *ValueType, llvm::IRBuilder<> &B,
                llvm::Align Alignment, llvm::AtomicOrdering LoadOrdering);

// Creates a typed store - either primitive or pointer store.
// For a primitive type stored value, this is just a store. Same for a pointer
// value stored in regular mode. In case of pointer typed stored value in
// compressed mode, this first does a compress and then a store.
llvm::StoreInst *CreateTypedStore(llvm::Value *ValueToWrite, llvm::Value *Addr,
                                  llvm::IRBuilder<> &B, llvm::Align Alignment,
                                  llvm::AtomicOrdering StoreOrdering);

// Return the unpoison function abstraction.
// ArgTy identifies what version of unpoison we need, scalar or a vector of VF.
llvm::Function *getUnpoisonFunc(llvm::Type *ArgTy, llvm::Module *M);

// Returns true if I is a safepoint poll call or a function call that may poll.
bool isLiteralOrImpliedPoll(llvm::Instruction *I);

// Return true if \p L has a finite trip count which is as specified by
// MaxWidth.
bool isKnownFiniteLoop(const llvm::Loop *L, unsigned MaxWidth,
                       llvm::ScalarEvolution *SE);

// Returns the chunked loop created from the OrigLoop if it is possible to do
// so. The chunked loop is run from 0 to ChunkSize iterations within OrigLoop.
// Optional HeaderSplitPoint and LatchSplitPoint decides at which instructions
// we want to
// create the branch into and out of the chunked loop (respectively).
// All instructions BEFORE and including HeaderSplitPoint, i.e. in range [
// HeaderBegin, HeaderSplitPoint ], will continue to be in the OrigLoop's
// header after chunking.
// Similarly, all instructions AFTER and including LatchSplitPoint, i.e. in
// range [ LatchSplitPoint, LatchEnd ], will continue to be in the OrigLoop's
// latch after chunking.
// In the case of a loop with exactly one block, we first need to split the loop
// into two blocks (header and latch). In such cases, we can pass in the
// SingleBlockLoopSplitPoint (optional), so that all instructions AFTER and
// including the SingleBlockLoopSplitPoint will belong to the latch.
std::optional<llvm::Loop *>
GenerateChunkedLoop(llvm::Loop *OrigLoop, llvm::LoopInfo *LI,
                    llvm::DominatorTree *DT, llvm::ScalarEvolution *SE,
                    uint64_t ChunkSize,
                    llvm::Instruction *HeaderSplitPoint = nullptr,
                    llvm::Instruction *LatchSplitPoint = nullptr,
                    llvm::Instruction *SingleBlockLoopSplitPoint = nullptr);

void GenerateMemfillBody(llvm::Function *F, llvm::Value *Dst, llvm::Value *Val,
                         llvm::Value *Len);

// This API just modifies OrigLoop to add a ChunkedIV and chunked condition to the
// existing loop. Unlike GenerateChunkedLoop, there is no change in the loop CFG
// (no added/deleted blocks), except for change in the LoopTripCount, where we
// now have MaxTripCount as min(OrigMaxTripCount, ChunkSize).
bool IntroduceChunkCondForLoop(llvm::Loop *OrigLoop, uint64_t ChunkSize,
                               llvm::ScalarEvolution *SE);

// Returns true if the LVB can potentially have an implied use.
// An implied use is one where the LVB is used as the "fixed up" value for
// another load from the same address as the LVB.
bool mayHaveImpliedUse(const llvm::CallInst &LVBCall);

// If BI is a LVB check, then return a condition which checks the value of the
// trap mask. This is the core logic for LVB unswitching.
// We also set matchedLVBCheck to true if BI was an LVB check. This is
// orthogonal to the Value that is returned by this function because
// matchedLVBCheck just states whether there was an LVB in the loop (whereas the
// value returned depends on many more conditions).
llvm::Value *generateLVBTrapMaskCheck(llvm::BranchInst *BI,
                                      llvm::Loop *currentLoop,
                                      bool &matchedLVBCheck,
                                      bool UnswitchOverLVBTrace);

// Matches the trap mask and returns that as Value.
llvm::Value *MatchLVB(llvm::BranchInst *BI, llvm::Loop *CurrentLoop);

// Returns true if load needs an LVB.
bool LoadNeedsLVB(const llvm::LoadInst &LI);

// Returns true if store needs an SVB.
bool StoreNeedsSVB(const llvm::StoreInst &SI);

// If there is a known value identifier associated with the given
// value, return the identifier. Else, return std::nullopt.
std::optional<uint64_t> getKnownValueID(const llvm::Value *V);

std::optional<uint64_t> getArgumentKnownValueID(llvm::CallBase *CB,
                                                unsigned ArgNo);
void setArgumentKnownValueID(llvm::CallBase *CB, unsigned ArgNo, uint64_t KVID);

/// Returns whether the given argument is marked as "unescaped before the call".
bool isArgumentUnescaped(const llvm::CallBase *CB, unsigned ArgNo);
/// Mark the given argument as "unescaped before the call".
void setArgumentUnescaped(llvm::CallBase *CB, unsigned ArgNo);

/// Returns whether the given allocation is marked as unescaped.
bool isUnescapedAllocation(const llvm::CallBase *CB);
/// Mark the given allocation as unescaped.
void markUnescapedAllocation(llvm::CallBase *CB);

/// Returns the number of allocations escaped throught this call.
std::optional<unsigned> getNumOfEscapedAllocations(const llvm::CallBase *CB);
/// Sets the number of allocations escaped throught this call.
void setNumOfEscapedAllocations(llvm::CallBase *CB, unsigned NumAllocations);

/// Downgrades the given monitorenter/monitorexit operation to a thread-local
/// version. It doesn't perform any analysis, just rewrites the operation. So,
/// it is callers responsibility to make sure that the use of a thread-local
/// operation is correct.  
void replaceMonitorOperationWithThreadLocal(llvm::CallBase *CB);

// Annotates the pointer type instruction with the given known-value ID.
// The instruction must always evaluate to a pointer to the given known value
// object and can never be null.
void setKnownValueID(llvm::Instruction *I, uint64_t KnownValueID);

std::optional<uint64_t> getArgumentKnownValueID(llvm::CallBase *CB,
                                                unsigned ArgNo);
void setArgumentKnownValueID(llvm::CallBase *CB, unsigned ArgNo, uint64_t KVID);

std::optional<uint64_t> getArgumentKnownValueID(llvm::Function *F,
                                                unsigned ArgNo);
void setArgumentKnownValueID(llvm::Function *F, unsigned ArgNo, uint64_t KVID);

using OptGetBFIFunc = std::optional<
    llvm::function_ref<llvm::BlockFrequencyInfo &(llvm::Function &)> >;
std::optional<double> getCallSiteFrequency(llvm::CallBase &Call,
                                           const OptGetBFIFunc &GetBFI);

// Accumulates the number of instructions inlined into the top level compile
// function of the given module in "azul.num.inlined.into.top.level" named
// metadata.
void IncrementNumInlinedInstrs(llvm::Module &F, uint64_t Num);

// Get amount of instructions we already inlined into the top level
// compile function of the given module. We use it in inline cost
// to prevent excessive inlining.
uint64_t GetNumInlinedInstrs(llvm::Module &F);

bool isAzulTopLevelMethod(const llvm::Function &F);
bool isJavaMethod(const llvm::Function &F);

bool isAtomicityOkayForVectorization(const llvm::LoadInst &LI);
bool isAtomicityOkayForVectorization(const llvm::StoreInst &SI);
bool disableGCPtrGathersScatters();
bool disableGCPtrMaskedLoadsStores();

// Apply a call stub provided by the VM to the given call site. The call stub
// will be inlined in place of the call site.
// If a non-null NewCallSites is passed it will be filled with the call sites
// introduced as the result of applying the call stub.
void ApplyCallStub(llvm::CallBase &CS, llvm::Function *Stub,
                   llvm::SmallVectorImpl<llvm::CallBase *> *NewCallSites =
                       nullptr);

// Extracts command line options from the 'azul.extra.opt.args' metadata
// and parses them via cl::ParseCommandLineOptions.
void ParseReplayCommandLineOptions(std::unique_ptr<llvm::Module> &M);

// Return true if 'I' stores (or loads) to the zero initialized part of the
// object 'Object'. Note that returned 'true' means that this location was zero
// initialized only during allocation of the object. This function doesn't check
// for interfering writes into same location.
bool isAccessToZeroInitializedLocation(const llvm::Instruction &I,
                                       const llvm::Instruction &Object,
                                       const llvm::TargetLibraryInfo *TLI);

// Returns true if LI is a load of the LVB trap mask, which is a global
// variable.
bool isLVBTrapMaskLoad(const llvm::LoadInst *LI);

// Returns true if V is a load of the LVB trap mask, which is a global
// variable.
bool isLVBTrapMaskLoad(const llvm::Value *V);

// Returns true if V is the LVB trap mask global variable.
bool isLVBTrapMask(const llvm::Value *V);

/// Returns true if the LVB trap mask might potentially be changed by the given
/// instruction. 
bool mayChangeLVBMask(const llvm::Instruction *I,
                      const llvm::TargetLibraryInfo &TLI);


constexpr const char *TrivialDeadAttributeName = "azul-trivial-deadness";
// Return True if call site is marked with azul-trivial-deadness attribute.
bool isAzulTrivialDead(const llvm::CallBase *CB);

/// Update the !loop metadata on all loops in F.
/// NB: currently just sets llvm.loop.unroll.runtime.multi.exit.enable
/// on all the loops.
/// TODO: we should refactor this interface to make it more obvious what
/// metadata is being updated.
void LoopMetadataUpdate(llvm::Function &F);
void LoopMetadataUpdate(llvm::Function &F, llvm::LoopInfo &LI);

/// Return true if any weight was changed.
bool NormalizeWeights(llvm::SmallVectorImpl<llvm::BranchWeightType> &Weights);

double RoundFrequency(double Frequency);
uint64_t RoundFunctionEntryCount(uint64_t Counter);

/// Normalize/coarsen the profile metadata in the given function.
/// Return true if any change was made.
bool NormalizeProfile(llvm::Function &F);

/// Drop all the profile metadata in the given function.
void DropProfile(llvm::Function &F, bool CallSiteOnly = false);

void AppendJVMStateCallers(std::optional<llvm::OperandBundleUse> &DeoptOB,
                           llvm::SmallVectorImpl<uint64_t> &CallerIDs, 
                           llvm::SmallVectorImpl<uint64_t> &CallerBCIs);
void AppendJVMStateCallers(llvm::CallBase *CB, 
                           llvm::SmallVectorImpl<uint64_t> &CallerIDs, 
                           llvm::SmallVectorImpl<uint64_t> &CallerBCIs);

/// Add back valid metadata from OrigCall to NewCall where OrigCall has a deopt
/// state.
void AddValidMetadataForDeoptCalls(const llvm::CallInst *OrigCall,
                                   llvm::CallInst *NewCall);

/// Combine the metadata of two instructions so that Dest can replace Src.
void combineSafepointMetadata(llvm::CallInst *Dest, const llvm::CallInst *Src);

/// Generates a clone of a given instruction with new deopt bundle.
llvm::Instruction *
cloneAndReplaceDeopt(llvm::CallBase *CB,
                     std::vector<llvm::Value *> &&NewDeoptArgs);

/// Accumulates transitive uses of the allocation (including uses of addresses 
/// derived from the allocation).
std::vector<llvm::Use *> collectTransitiveUses(llvm::Value *New);

template <typename T>
T GetOptionFromModuleOrGlobal(llvm::Module &M, llvm::cl::opt<T> &Opt) {
  using namespace llvm;
  T Result = Opt.getValue();

  if (NamedMDNode *NN = M.getNamedMetadata("azul.extra.opt.local.args"))
    for (auto *Arg : NN->operands()) {
      auto *ArgKey = cast<MDString>(Arg->getOperand(0));
      if (ArgKey->getString().equals(Opt.ArgStr)) {
        auto *ArgValue = cast<MDString>(Arg->getOperand(1));
        Opt.getParser().parse(Opt, Opt.ArgStr, ArgValue->getString(), Result);
        // Independent on parsing passed or not, return Result.
        // It will be the new Result in case of success or original one in case
        // of failure.
        break;
      }
    }
  return Result;
}

// Stores inlining information that gets unavailable after inlining but needed
// to be reported to VM when inlining is done.
class AzulInlineTracker {
  bool ReportJavaMethodInlining;
  llvm::Function *Caller;
  llvm::Function *Callee;
  llvm::SmallVector<uint64_t, 16> CallerIDs;
  llvm::SmallVector<uint64_t, 16> CallerBCIs;

public:
  AzulInlineTracker(llvm::CallBase &Call);

  void
  reportSuccessfulInline(const llvm::InlineFunctionInfo &InlineInfo,
                         llvm::StringRef Message = llvm::StringRef()) const;
};

bool isGeneratableAbstraction(const llvm::Function *F);
bool CanGenerateCallee(const llvm::CallBase& Call);

/// Replace function ID of the given function. This includes changing the 
/// function attribute and the caller ID in the root frame in the deopt states 
/// inside of this function.
///
/// To be used when a specialized copy is created. A new function ID for the
/// specialized copy should be provided by the VM. This function can only be 
/// used for definitions. It doesn't make much sense to specialize a function 
/// without a body.
void ReplaceFunctionID(llvm::Function *F, uint32_t NewFunctionID);

/// Creates a specialized copy of the function.
std::optional<llvm::Function *> createFunctionSpecialization(llvm::Function *F);

/// Creates a call-site specialized callee copy for the given call site if the
/// callee is not already specialized for this call-site.
/// Returns the call-site specialized callee on success.
std::optional<llvm::Function *>
createCallSiteSpecialization(llvm::CallBase *Call);

/// Return true two functions correspond to the same Java method.
std::optional<bool> isSameMethod(llvm::LLVMContext &C, uint64_t AFuncID,
                                 uint64_t BFuncID);
std::optional<bool> isSameMethod(llvm::LLVMContext &C, llvm::Function *A,
                                 llvm::Function *B);

/// \brief Returns number of occurrences of the callee function in the inline
/// call stack of the specified call site excluding the call site itself.
/// Zero corresponds to non-recursive callsite.
///
/// We use deopt operand bundles in order to detect recursion. In case there
/// is no relevant operand bundles we return std::nullopt.
std::optional<unsigned>
JavaRecursionDepth(llvm::ArrayRef<const llvm::CallBase *> CallStack,
                   const llvm::Function &F);
std::optional<unsigned> JavaRecursionDepth(const llvm::CallBase &CB,
                                           const llvm::Function &F);

// Call replaceAllUsesWith() and eraseFromParent() but keeps the old name
// intact.
void replaceAllUsesAndErase(llvm::Instruction *Old, llvm::Value *New);

/// Updates data layout setting incorporating downstream changes.
std::string computeDataLayout(std::string DL);
} // End Utils namespace

template <typename T>
using BumpPtrAllocatorDeleterType = std::function<void(T *)>;

template <typename T>
using UniqueBumpPtrAllocatorPtr =
    std::unique_ptr<T, BumpPtrAllocatorDeleterType<T>>;

// Allocates memory using BumpPtrAllocator for a new object of type T, calls a
// constructor with the given args and returns a std::unique_ptr containing that
// object with a custom deleter that just calls the destructor of the  object.
// No memory is freed by the deleter. All the memory allocated by the
// BumpPtrAllocator is expected to be cleared by the user manually.
template <typename T, typename... CtorArgs>
UniqueBumpPtrAllocatorPtr<T>
make_unique_bump_ptr_alloc(llvm::BumpPtrAllocator &Allocator,
                           CtorArgs &&...Args) {
  auto Obj = new (Allocator.Allocate<T>()) T(std::forward<CtorArgs>(Args)...);
  auto Deleter = [&](T *Ptr) {
    Ptr->~T();
    Allocator.Deallocate(Ptr);
  };
  return UniqueBumpPtrAllocatorPtr<T>(Obj, Deleter);
}

bool isArrayLengthOffset(llvm::LLVMContext &C, int64_t Offset);
bool isObjArrayElementKidOffset(llvm::LLVMContext &C, int64_t Offset);

/// NOTE: This function can *only* be used for assessing *profitability* of a
/// transformation, not correctness thereof.  It doesn't provide extent
/// information, so using it for legality would break if we widden loads.
bool isLVBTrapMaskAddr(llvm::LLVMContext &C, uint64_t AddrMaybe);
  
/// Represents a node in the inline tree recreated from debug metadata.
struct InlinedTreeNode {
  unsigned Line = 0;
  unsigned Column = 0;
  llvm::DILocalScope *Scope = nullptr;

  // DILocation metadata nodes matching with this InlinedTreeNode
  llvm::SmallPtrSet<llvm::DILocation *, 8> DILocations;
  llvm::SmallVector<InlinedTreeNode *, 20> Callees;

  InlinedTreeNode() {} // Root node constructor
  InlinedTreeNode(llvm::DILocation *DI);
  ~InlinedTreeNode() {
    for (auto *Node : Callees)
      delete (Node);
    return;
  }

  bool matches(llvm::DILocation *Loc);
  void print(const llvm::Module *M, int Indent,
             llvm::formatted_raw_ostream &OS);
  InlinedTreeNode *findByDILocation(llvm::DILocation *Loc);
};

/// Collects the inline tree for the given function by looking at the debug
/// metadata on the instructions.
InlinedTreeNode *collectInlineTree(const llvm::Function *F);

/// Pretty prints Orca-specific debug location info. 
void printDebugLocation(llvm::StringRef Prefix, llvm::DILocation *DIL,
                        llvm::formatted_raw_ostream &OS);

/// Pretty prints Orca-specific debug info for the given instruction.
void printDebugInfo(llvm::StringRef Prefix, const llvm::Instruction *I, 
                    llvm::formatted_raw_ostream &OS);
/// Pretty prints common metadata nodes for the given instruction.
void printMetadata(llvm::StringRef Prefix, const llvm::Instruction *I, 
                   llvm::formatted_raw_ostream &OS);

/// Returns true if the value is a box object.
/// A box object is a wrapper over a primitive value produced as a
/// result of a call to the corresponding valueOf method.
bool isBoxObject(const llvm::Value *V);

/// ValueInfo is an aggregate to hold facts we know about a value.
struct ValueInfo {
  /// The LLVM type of the value.
  /// Some of the ValueInfo fields are applicable only to GC pointers. 
  llvm::Type *Ty = nullptr;

  /// Set if the value is a compile time constant.
  ///
  /// Note that a GC pointer value can be a compile time constant, but the only
  /// valid values are UndefValue and NullValue.
  std::optional<llvm::Constant *> ConstantValue;
  /// KnownValueID for the value. Makes sense only for GC pointer values.
  std::optional<uint64_t> KnownValueID;
  /// JavaType of the value. Makes sense only for GC pointer values.
  std::optional<TypeUtils::JavaType> JType;

  ValueInfo() {}
  ValueInfo(llvm::Type *Ty) : Ty(Ty) {}
  ValueInfo(llvm::Type *Ty, std::optional<llvm::Constant *> ConstantValue,
            std::optional<uint64_t> KnownValueID,
            std::optional<TypeUtils::JavaType> JType);
  ValueInfo(const ValueInfo &Other) = default;
  ValueInfo& operator=(const ValueInfo &Other) = default;

  bool operator==(const ValueInfo &Other) const {
    return Ty == Other.Ty &&
           ConstantValue == Other.ConstantValue &&
           KnownValueID == Other.KnownValueID &&
           JType == Other.JType;
  }

  /// Combines facts about the same value coming from different sources.
  /// If the given facts are incompatible we are analyzing dead code and can
  /// return any value.
  ValueInfo joinWith(const ValueInfo &Other);
  /// Merges facts about values coming from different paths.
  ValueInfo meetWith(const ValueInfo &Other);
  bool isMoreSpecificThan(const ValueInfo &Other);

  void dump();
  void print(llvm::raw_ostream &OS) const;
};

inline llvm::raw_ostream &operator<<(llvm::raw_ostream &OS,
                                     const ValueInfo &VI) {
  VI.print(OS);
  return OS;
}
ValueInfo getArgumentValueInfo(llvm::CallBase *CB, unsigned ArgNo,
                               llvm::DominatorTree *DT = nullptr);
ValueInfo getValueInfo(llvm::Value *V, const llvm::Instruction *CtxI = nullptr,
                       const llvm::DominatorTree *DT = nullptr);

/// If the function \p F effectively returns one single value, returns it. E.g.
/// it tries to look through PHINodes and ignores returns from @llvm.deoptimize.
/// If the function returns several different values, returns std::nullopt.
std::optional<llvm::Value *> getReturnedValue(llvm::Function *F);

/// A convenience wrapper over the VMInterface::getConstantResult which returns
/// the constant result for the given call site.
std::optional<uint64_t> getConstantResult(llvm::CallBase &Call,
                                          llvm::DominatorTree *DT = nullptr);

/// A convenience wrapper over the VMInterface::getSpecializedImplementation
/// which returns the specialized implementation for the given call site.
std::optional<llvm::Function *>
getSpecializedImplementation(llvm::CallBase &Call, llvm::Module &M,
                             llvm::DominatorTree *DT = nullptr);

/// Helper function to read "azul.VP" metadata which has the following format:
///   !prof !{!"azul.VP",
///          [i32 OpNum, !{i64 TotalWeight, [i64 Value, i64 Weight]+}]*}
///
/// First it searches for the data related to operand number \p OpNum of \p Inst
/// (if any) and then pushes it into \p ValueProfMDVec vector. As a result,
/// 'TotalWeight' will be the first element of \p ValueProfMDVec followed by
/// '{Value, Weight}' pairs.
/// Returns true if profiling data for the specified operand exists, false
/// otherwise.
// TODO: Consider upstreaming this profiling metadata. It should substitute
// existing VP metadata.
bool getAzulValueProf(const llvm::Instruction *Inst, uint32_t OpNum,
                      llvm::SmallVectorImpl<uint64_t> &ValueProfMDVec);

/// Removes "azul.VP" profiling metada for operand number \p OpNum of \p Inst.
/// Returns true if data has been successfully removed, false otherwise.
bool removeAzulValueProf(llvm::Instruction *Inst, uint32_t OpNum);

/// Use this function if you have a graph printer which works on individual
/// functions and you want it to have the same controls as other graph printers.
bool shouldDumpFunctionGraph(const llvm::Function &F);

/// The key difference from llvm::TimeRecord is that it uses integers to store
/// time instead of doubles which means it should work better for accumulating
/// time measurements, especially smaller ones. Stores time in microseconds.
class OrcaTimeRecord {
  uint64_t WallTime = 0;
  uint64_t UserTime = 0;
  uint64_t SystemTime = 0;

public:
  OrcaTimeRecord() = default;

  static OrcaTimeRecord getCurrentTime();
  static OrcaTimeRecord getTimeSince(const OrcaTimeRecord &Start) {
    return getCurrentTime() - Start;
  }

  uint64_t getUserTime() const { return UserTime; }
  uint64_t getSystemTime() const { return SystemTime; }
  uint64_t getThreadTime() const { return UserTime + SystemTime; }
  uint64_t getWallTime() const { return WallTime; }

  OrcaTimeRecord &operator+=(const OrcaTimeRecord &Other) {
    WallTime += Other.WallTime;
    UserTime += Other.UserTime;
    SystemTime += Other.SystemTime;
    return *this;
  }

  OrcaTimeRecord &operator-=(const OrcaTimeRecord &Other) {
    WallTime -= Other.WallTime;
    UserTime -= Other.UserTime;
    SystemTime -= Other.SystemTime;
    return *this;
  }

  friend OrcaTimeRecord operator+(const OrcaTimeRecord &LHS,
                                  const OrcaTimeRecord &RHS) {
    OrcaTimeRecord Result = LHS;
    Result += RHS;
    return Result;
  }

  friend OrcaTimeRecord operator-(const OrcaTimeRecord &LHS,
                                  const OrcaTimeRecord &RHS) {
    OrcaTimeRecord Result = LHS;
    Result -= RHS;
    return Result;
  }

  void print(llvm::raw_ostream &OS);
};
} // End azul namespace

#endif
