//===- llvm/Orca/AzulSink.h - Definition of the AzulSink class --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
// 
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This file provides the interface for Azul's own Azul Sink pass.
///
//===----------------------------------------------------------------------===//

#ifndef ORCA_SINK_H
#define ORCA_SINK_H

#include "llvm/Analysis/CFG.h"
#include "llvm/Analysis/LoopInfo.h"
#include "llvm/Analysis/Orca/FlowSensitiveEA.h"
#include "llvm/IR/Dominators.h"
#include "llvm/IR/PassManager.h"
#include "llvm/Orca/FlowSensitiveEATransform.h"

namespace llvm {

class Function;

struct AzulSink : PassInfoMixin<AzulSink> {

public:
  AzulSink() = default;

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &MAM);
};

struct AzulSinkBarriers : PassInfoMixin<AzulSinkBarriers> {

public:
  AzulSinkBarriers() = default;

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &MAM);
};

} // end namespace llvm

namespace azul::FlowSensitiveEA {

using namespace llvm;

class SinkPublicationBarriers {
  Function &PublicationBarrier;
  SmallVectorImpl<Transform *> &Transforms;
  DominatorTree &DT;
  LoopInfo &LI;

  struct RemovePublicationBarrier : public Erase {
    bool Canceled = false;

    RemovePublicationBarrier(Instruction *ToRemove) : Erase(ToRemove) {}
    ~RemovePublicationBarrier() {}

    void cancel() { Canceled = true; }

    IRChange apply() override {
      return Canceled ? IRChange::None : Erase::apply();
    }
  };

  // For each allocation remember the last RemovePublicationBarrier transform.
  // This transformation will be cancelled if there is an
  // InsertPublicationBarrier transformatin later in this block for the same
  // allocation.
  DenseMap<Instruction *, RemovePublicationBarrier *> LastRemovedBarriers;

  struct InsertPublicationBarrier : public Transform {
    SinkPublicationBarriers &SPB;
    TrackingVH<Instruction> Allocation;
    TrackingVH<Instruction> InsertBefore;

    InsertPublicationBarrier(SinkPublicationBarriers &SPB, Instruction *Alloc,
                             Instruction *InsertBefore)
        : SPB(SPB), Allocation(Alloc), InsertBefore(InsertBefore) {
      assert(Alloc);
      assert(InsertBefore);
    }
    ~InsertPublicationBarrier() {}

    IRChange apply() override {
      return SPB.insertPublicationBarrierBefore(Allocation.getValPtr(),
                                                InsertBefore.getValPtr());
    }
  };

  struct SplitEdgeAndInsertPublicationBarriers : public Transform {
    SinkPublicationBarriers &SPB;
    SmallVector<TrackingVH<Instruction>, 4> Allocations;
    // The pair (From, SuccNum) specifies the edge to split. Instead of
    // BasicBlock the number SuccNum is used to tolerate CFG changes (edge
    // splitting).
    AssertingVH<BasicBlock> From;
    unsigned SuccNum;

    SplitEdgeAndInsertPublicationBarriers(
        SinkPublicationBarriers &SPB, SmallVectorImpl<Instruction *> &Allocs,
        BasicBlock *From, BasicBlock *To)
        : SPB(SPB), From(From), SuccNum(GetSuccessorNumber(From, To)) {
      for (auto *V : Allocs)
        Allocations.emplace_back(V);
    }
    ~SplitEdgeAndInsertPublicationBarriers() {}

    IRChange apply() override {
      return SPB.splitEdgeAndInsertPublicationBarriers(Allocations, From,
                                                       SuccNum);
    }
  };

  Transform::IRChange splitEdgeAndInsertPublicationBarriers(
      SmallVectorImpl<TrackingVH<Instruction>> &Allocations, BasicBlock *From,
      unsigned SuccNum);

  Transform::IRChange insertPublicationBarrierBefore(Instruction *Allocation,
                                                     Instruction *InsertBefore);

  bool createInsertPublicationBarrier(Instruction *Alloc,
                                      Instruction *InsertBefore);

  bool createRemovePublicationBarrierTransform(const State &S, CallBase *CB);

public:
  explicit SinkPublicationBarriers(Function &PublicationBarrier,
                                   SmallVectorImpl<Transform *> &Transforms,
                                   Module &M, DominatorTree &DT, LoopInfo &LI)
      : PublicationBarrier(PublicationBarrier), Transforms(Transforms), DT(DT),
        LI(LI){};

  // Collect transforms while visiting instructions.
  void visitInstruction(Instruction &I, bool StateChanged, State &PrevState,
                        State &S);

  // Collect transforms while visiting end of basic blocks.
  void visitEndOfBlock(BasicBlock &BB, FlowSensitiveEscapeAnalysis &EA);
};

} // end namespace azul::FlowSensitiveEA
#endif // ORCA_SINK_H
