//===-- llvm/Orca/OrcaPassTimer.h -------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2022 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header file defines the pass timer wrapper which can be used to measure
// execution time of any pass in pipeline.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_ORCA_PASS_TIMER
#define LLVM_ORCA_PASS_TIMER

#include "llvm/ADT/StringRef.h"
#include "llvm/Analysis/LoopAnalysisManager.h"
#include "llvm/IR/LLVMContext.h"
#include "llvm/IR/PassManager.h"
#include "llvm/Orca/OrcaPassManagers.h"
#include "llvm/Orca/OrcaState.h"
#include "llvm/Transforms/Scalar/LoopPassManager.h"
#include <string>

namespace llvm {

namespace {
template <typename IRUnitT>
LLVMContext &getContext(IRUnitT &IRUnit) {
  return IRUnit.getContext();
}

// llvm::Loop class doesn't have 'getContext' method, so as a workaround we
// get it from the loop's header block.
template <>
LLVMContext &getContext(Loop &L) {
  return L.getHeader()->getContext();
}
} // namespace

template <typename PassT>
class OrcaPassTimer
    : public PassInfoMixin<OrcaPassTimer<PassT>> {
  PassT Pass;
  std::string TimerName;

public:
  OrcaPassTimer(PassT &&Pass, StringRef TimerName)
      : Pass(std::forward<PassT>(Pass)), TimerName(TimerName) {}

  template <typename IRUnitT,
            typename AnalysisManagerT = AnalysisManager<IRUnitT>,
            typename... ExtraArgTs>
  PreservedAnalyses run(IRUnitT &IRUnit, AnalysisManagerT &AM,
                        ExtraArgTs &&...ExtraArgs) {
    LLVMContext &C = getContext(IRUnit);
    azul::TimeRegion T(C, TimerName);
    // No need to update AM as we execute only one pass, invalidation
    // will be done by caller.
    return azul::internal::runOnePass(Pass, IRUnit, AM, false /* updateAM */,
                                      std::forward<ExtraArgTs>(ExtraArgs)...);
  }
};

template <typename PassT>
OrcaPassTimer<PassT> createPassTimer(PassT &&Pass, StringRef TimerName) {
  return OrcaPassTimer<PassT>(std::forward<PassT>(Pass), TimerName);
}

} // namespace llvm

#endif
