//===-- OrcaFeatures.h ------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2019 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header defines the API that exposes Orca compilation Features.
//
//===----------------------------------------------------------------------===//
#ifndef LLVM_AZUL_ORCA_FEATURES_H
#define LLVM_AZUL_ORCA_FEATURES_H

#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/DenseSet.h"
#include "llvm/ADT/ArrayRef.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/ADT/SparseBitVector.h"
#include "llvm/ADT/StringRef.h"
#include "llvm/Support/Debug.h"
#include <optional>

namespace llvm {
  class Module;
}

namespace azul {
namespace orca {

enum class SimplifyWithInlining { No, Trivial, Full };

/// Enumerator for individual features.
/// Refer to OrcaFeatures.def file for actual definition of names.
enum class OrcaFeatureID : unsigned {
#define GLOBAL_FEATURE(NAME, TYPE) NAME,
#define LOCAL_FEATURE(NAME, TYPE) NAME,
#include "OrcaFeatures.def"
};

struct OrcaFeatureTraits {
public:
  /// A helper to redefine type/value for individual features.
  template <OrcaFeatureID> struct Impl {
    /// A type of value for corresponding OrcaFeature made available for
    /// compile-time checks.
    using ValueType = void;

    /// Initialize as true for global features, false for local features.
    static bool IsGlobal;
  };
};

/// This class provides external Orca users the interface for compilation
/// features hiding majority of implementation details of any particular
/// feature.
///
/// Object of this class represents a set of values for enabled compilation
/// features. Use it to "apply" compilation features either to LLVM instance
/// (one-time initialization of globals for llvm shared library), to a
/// particular Module to save them for replay compilations or even to a
/// particular compilation to fine-tune compiler behavior.
///
/// Note, that individual features can be either Local or Global, which is
/// mainly defined by details of their implementation.
/// Local features can be applied to a particular compilation, even if there are
/// many compilations running in parallel.
/// Global features can only be applied to LLVM instance as a whole, meaning
/// that practically you can only do global features application at the time of
/// LLVM initialization - any changes you do globally will affect all the
/// compilations running in parallel.
///
/// The main purpose of Global features is to make transition of flags to Local
/// features smooth. So, if you don't think we'll ever want to convert this
/// feature into Local, you don't have to add a Global feature for your flag.
class OrcaFeatures {
public:
  template <OrcaFeatureID ID>
  using ValueType = typename OrcaFeatureTraits::Impl<ID>::ValueType;

private:
  /// value fields <FeatureName>Value
#define GLOBAL_FEATURE(NAME, TYPE) std::optional<TYPE> NAME##Value;
#define LOCAL_FEATURE(NAME, TYPE) GLOBAL_FEATURE(NAME, TYPE)

#include "OrcaFeatures.def"

public:
  OrcaFeatures() = default;

  /// A convenient way to get a reference to an empty OrcaFeatures object with
  /// static lifetime.
  static const OrcaFeatures &getEmpty() { return NoFeatures; }

  /// This method sets only local features, it will crash if a global
  /// feature provided.
  template <OrcaFeatureID ID> void enableLocal(ValueType<ID> Value) {
    static_assert(!OrcaFeatureTraits::Impl<ID>::IsGlobal,
                  "Only local features allowed!");
    enable<ID>(Value);
  }

  void collectGlobalLLVMFlags(std::vector<std::string>& LLVMFlags) const;

  /// This method is intended to be used for saving all local flags in the
  /// module for IR dumps.
  void applyLocalLLVMFlags(llvm::Module &M) const;

  template <OrcaFeatureID ID> std::optional<ValueType<ID>> getValue() const;

  template <OrcaFeatureID ID>
  ValueType<ID> getValueOr(ValueType<ID> Default) const {
    return getValue<ID>().value_or(Default);
  }

  /// \p Str should contain one or more OrcaFeature setting, each of them
  /// should be in one of two supported formats:
  ///   1) LocalOrcaFeatureID=FeatureValue
  ///   2) +LocalOrcaFeatureID or -LocalOrcaFeatureID
  /// If \p Error is not nullptr and parsing failed, this string will contain
  /// the error message, otherwise it will be left intact.
  /// Returns true if parsed and applied all features successfully, false
  /// otherwise.
  bool parseAndApplyLocalFeatures(llvm::StringRef Str);

  /// Dry run of parseAndApplyLocalFeatures, does the same thing, except that it
  /// doesn't apply the flags.
  static bool verifyLocalFeatures(llvm::StringRef Str, std::string &Error);

protected:
  // This method can set both local and global features.
  template <OrcaFeatureID ID> void enable(ValueType<ID> Value);

private:
  void collectLocalLLVMFlags(std::vector<std::string> &LLVMFlags) const;

  void applyFlag(std::vector<std::string>& Args, OrcaFeatureID ID,
                 unsigned Value) const;

  /// utility to use in templated static_asserts
  template <OrcaFeatureID...> static constexpr bool false_v = false;

public:
  // Individual explicit specializations should provide implementation, even if
  // empty. If you get static_assert here it means you forgot to define explicit
  // specialization for your new feature.
  template <OrcaFeatureID ID>
  void applyFlag(std::vector<std::string> &Args, ValueType<ID> Value) const {
    static_assert(false_v<ID>, "Please, define explicit specialization of "
                               "applyFlag for your FeatureID, even if empty. "
                               "There is no default implementation.");
  }

  static const OrcaFeatures NoFeatures;
};

///
/// Defining majority of feature properties - type, global/local, value
/// accessors. We use preprocessor magic of OrcaFeatures.def to "iterate"
/// through all OrcaFeatureID enum values.
///
#define DefineFeature(NAME, TYPE, GLOB)                                        \
  template <> struct OrcaFeatureTraits::Impl<OrcaFeatureID::NAME> {            \
    using ValueType = TYPE;                                                    \
    static constexpr bool IsGlobal = GLOB;                                     \
  };                                                                           \
                                                                               \
  template <>                                                                  \
  inline std::optional<TYPE> OrcaFeatures::getValue<OrcaFeatureID::NAME>()     \
      const {                                                                  \
    return NAME##Value;                                                        \
  }                                                                            \
  template <>                                                                  \
  inline void OrcaFeatures::enable<OrcaFeatureID::NAME>(TYPE Value) {          \
    NAME##Value = Value;                                                       \
  }

#define GLOBAL_FEATURE(NAME, TYPE) DefineFeature(NAME, TYPE, true)
#define LOCAL_FEATURE(NAME, TYPE) DefineFeature(NAME, TYPE, false)
#include "OrcaFeatures.def"
#undef DefineFeature

/// This enum defines Orca optimization level: it translates into a set of
/// OrcaFeatures (see CommonOrcaFeatures class below).
enum class OrcaOptLevel : unsigned {
  O0,
  O1,
  O2,
  O3,
  O4,
};

const char* getOrcaOptLevelName(OrcaOptLevel level);

/// This class holds a set of features that are common for all compilations.
/// Note that this class keeps cache of OrcaFeatures for different optimization
/// levels and the cache is not thread-safe, so instances of this class are
/// intended to be thread-local.
class CommonOrcaFeatures {
public:
  CommonOrcaFeatures() = default;

  CommonOrcaFeatures(
      const OrcaFeatures &CommonFeatures,
      llvm::ArrayRef<OrcaFeatureID> ExplicitValues,
      llvm::ArrayRef<std::string> ExtraCommonGlobalLLVMFlags = std::nullopt);

  /// Returns a reference to a set of Orca features configured for the provided
  /// opt level.
  /// Lifetime of a returned reference is equal to lifetime of the cache itself.
  const OrcaFeatures &getWithOptLevel(OrcaOptLevel OptLevel);

  llvm::ArrayRef<std::string> getGlobalLLVMFlags() const {
    return Common.getGlobalLLVMFlags();
  }

  /// Stores values of all global flags to the module (for replay compilations).
  void applyGlobalLLVMFlags(
      llvm::Module &M,
      llvm::ArrayRef<std::string> ExtraLLVMFlags = std::nullopt) const;

  /// One-time initialization of LLVM shared-library instance.
  void initializeGlobalLLVMFlags() const;

  /// Dump contents of common orca features set.
  void dump(llvm::raw_ostream& = llvm::dbgs()) const;

private:
  llvm::DenseMap<unsigned /*OrcaOptLevel*/, OrcaFeatures> FeaturesCache;

  /// A set of features and other misc information common for all optimization
  /// levels.
  class {
    OrcaFeatures Features;

    /// A set of features explicitly requested by user, createWithOptLevel
    /// should not change values for these features from a common feature set.
    llvm::SparseBitVector<> Explicit;

    /// This field holds combination of extra general global flags passed to the
    /// constructor and global flags implied by CommonFeatures.
    std::vector<std::string> GlobalLLVMFlags;

  public:
    void initialize(const OrcaFeatures &InitFeatures,
                    llvm::ArrayRef<OrcaFeatureID> ExplicitValues,
                    llvm::ArrayRef<std::string> ExtraGeneralGlobalLLVMFlags) {
      Features = InitFeatures;
      for (OrcaFeatureID ID : ExplicitValues)
        Explicit.set(static_cast<unsigned>(ID));

      auto GetFlagName = [](llvm::StringRef Flag) {
        size_t Begin = Flag.find_first_not_of('-');
        size_t End = Flag.find('=');
        return Flag.slice(Begin, End);
      };
      // Here we allow flags from ExtraGeneralGlobalLLVMFlags override the flags
      // we get from Features.collectGlobalLLVMFlags. While doing so, we
      // intentionally don't deduplicate flags within those sets to avoid hiding
      // cases when we have conflicting options in either of them (e.g. two
      // OrcaFeatures that set the same flag).
      llvm::DenseSet<llvm::StringRef> AddedExtraFlags;
      for (auto &Flag : ExtraGeneralGlobalLLVMFlags) {
        AddedExtraFlags.insert(GetFlagName(Flag));
        GlobalLLVMFlags.push_back(Flag);
      }
      std::vector<std::string> FeaturesFlags;
      Features.collectGlobalLLVMFlags(FeaturesFlags);
      for (auto &Flag : FeaturesFlags)
        if (!AddedExtraFlags.count(GetFlagName(Flag)))
          GlobalLLVMFlags.push_back(Flag);
    }

    llvm::ArrayRef<std::string> getGlobalLLVMFlags() const {
      return GlobalLLVMFlags;
    }

    /// Returns true if a given feature is explicitly requested by user.
    bool isExplicit(OrcaFeatureID ID) const {
      return Explicit.test(static_cast<unsigned>(ID));
    }

    OrcaFeatures getFeatures() const { return Features; }
  } Common;

  /// Enables feature value only if it not explicitly requested by user.
  /// To be used in OptLevel handling.
  template <OrcaFeatureID ID>
  void enableNonExplicit(OrcaFeatures &Features,
                         OrcaFeatures::ValueType<ID> Value) const;

  /// Returns a full set of features corresponding to OptLevel,
  /// with explicit settings on top.
  OrcaFeatures createWithOptLevel(OrcaOptLevel OptLevel) const;
};

/// This class should be used to construct CommonOrcaFeatures which is
/// immutable after construction. Only during construction of
/// OrcaCommonFeatures (i.e. through this class) you can set global features.
class CommonOrcaFeaturesBuilder : private OrcaFeatures {
  /// A set of features explicitly requested by user, opt-level should not
  /// change values for these features from a common feature set.
  llvm::SmallVector<OrcaFeatureID, 8> ExplicitValues;

public:
  /// This method allows to set both local and global features.
  /// Besides setting the feature value it also notes if particular feature
  /// value has been explicitly requested by user or not (explicitly requested
  /// values will not be overriden by per-compilation OptLevel features).
  template <OrcaFeatureID ID>
  void enable(ValueType<ID> Value, bool Explicit = false) {
    if (Explicit)
      ExplicitValues.push_back(ID);
    OrcaFeatures::enable<ID>(Value);
  }

  /// This is a temporary workaround introduced as a transitional measure to
  /// keep Zing interface of setting SimplifyCandidatesWithInlining stable while
  /// changing its type from bool to enum. Trades some temporary busy work in
  /// Orca for less changes and dependencies in Zing.
  template <OrcaFeatureID ID,
            typename = std::enable_if_t<
                ID == OrcaFeatureID::SimplifyCandidatesWithInlining>>
  void enable(bool Value, bool Explicit) {
    if (Explicit)
      ExplicitValues.push_back(ID);
    OrcaFeatures::enable<ID>(Value ? SimplifyWithInlining::Full
                                   : SimplifyWithInlining::No);
    ;
  }

  /// \param ExtraGlobalLLVMFlags The optional list of extra LLVM flags that
  /// should be common for all compilations.
  CommonOrcaFeatures
  build(llvm::ArrayRef<std::string> ExtraGlobalLLVMFlags = std::nullopt) const {
    return {*this, ExplicitValues, ExtraGlobalLLVMFlags};
  }
};

}
}
#endif
