//===---- AzulAliasAnalysis.h - Azul Alias Analysis -------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
//  This file contains azul specific alias analysis. It's a collection
//  of aliasing rules not expressable via normal llvm ir.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_ALIAS_ANALYSIS
#define LLVM_AZUL_ALIAS_ANALYSIS

#include "llvm/Analysis/AliasAnalysis.h"

namespace azul {
namespace TypeUtils {
class JavaType;
}

namespace TypeBasedAzulAA {

/// For the given azul-base-pointer-java-type metadata node, extracts the
/// corresponding JavaType.
TypeUtils::JavaType getBasePointerJavaType(llvm::MDNode *MDNode);

/// Creates an azul-base-pointer-java-type metadata node for the given JavaType
llvm::MDNode *createBasePointerJavaTypeMDNode(llvm::LLVMContext &C,
                                              TypeUtils::JavaType T);
}
}

namespace llvm {

class DataLayout;
class TargetLibraryInfo;
class MemoryLocation;

class AzulAAResult : public AAResultBase {

  const DataLayout &DL;
  std::function<const TargetLibraryInfo &(Function &F)> GetTLI;

  /// Return true if we could prove that aliasing between \p Call
  /// and \p Loc is Ref. Otherwise, return false.
  bool isRefOnly(const CallBase *Call, const MemoryLocation &Loc,
                 AAQueryInfo &AAQI) const;

public:
  explicit AzulAAResult(
      const DataLayout &DL,
      std::function<const TargetLibraryInfo &(Function &F)> GetTLI)
      : DL(DL), GetTLI(std::move(GetTLI)) {}
  AzulAAResult(AzulAAResult &&Arg)
      : AAResultBase(std::move(Arg)), DL(Arg.DL),
        GetTLI(std::move(Arg.GetTLI)) {}

  /// Handle invalidation events from the new pass manager's function analysis
  /// manager.
  ///
  /// This result is stateless by definition, also it has no stateful
  /// dependencies. Thus it remains valid all the time.
  bool invalidate(Function &F, const PreservedAnalyses &,
                  FunctionAnalysisManager::Invalidator &) {
    return false;
  }

  ModRefInfo getModRefInfoMask(const MemoryLocation &Loc,
                              AAQueryInfo &AAQI, bool IgnoreLocals);

  AliasResult alias(const MemoryLocation &LocA, const MemoryLocation &LocB,
                    AAQueryInfo &AAQI, const Instruction *I = nullptr);
  ModRefInfo getModRefInfo(const CallBase *Call, const MemoryLocation &Loc,
                           AAQueryInfo &AAQI);
  ModRefInfo getModRefInfo(const CallBase *Call1, const CallBase *Call2,
                           AAQueryInfo &AAQI);
};

/// Analysis pass providing a never-invalidated alias analysis result.
class AzulAA : public AnalysisInfoMixin<AzulAA> {
  friend AnalysisInfoMixin<AzulAA>;
  static AnalysisKey Key;

public:
  using Result = AzulAAResult;

  AzulAAResult run(Function &F, FunctionAnalysisManager &AM);
};

/// Legacy wrapper pass to provide the AzulAAResult object.
class AzulAAWrapperPass : public ImmutablePass {
  std::unique_ptr<AzulAAResult> Result;

public:
  static char ID;

  AzulAAWrapperPass();

  AzulAAResult &getResult() { return *Result; }
  const AzulAAResult &getResult() const { return *Result; }

  bool doInitialization(Module &M) override;
  bool doFinalization(Module &M) override;
  void getAnalysisUsage(AnalysisUsage &AU) const override;
};

}

#endif // LLVM_AZUL_ALIAS_ANALYSIS
