
//===-- Orca.h ---------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header defines the main API exposed by Orca.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_ORCA_H
#define LLVM_AZUL_ORCA_H

#include "llvm/ADT/ArrayRef.h"
#include "llvm/Orca/OrcaPipeline.h"
#include "llvm/Orca/VMInterface.h"
#include "llvm/Passes/PassBuilder.h"
#include "llvm/Support/Orca/VMCallbacks.h"

#include <optional>

namespace llvm {
namespace json {
  class OStream;
}
}

namespace azul {
namespace orca {

extern cl::opt<bool> PrintOrcaPipelinePasses;
extern cl::opt<bool> DebugOrcaPM;

/// To make a class `C` cacheable by the query cache, you need to:
///
///  - Implement a traverseFields method that tell the query cache about the
///    various fields in `C` it needs to be aware of.  See below for examples.
///
///  - Add a default constructor.  If you want to make that constructor private
///    (and not part of the object's public API), then add a `friend struct
///    Cacheable<C>` declaration.
///
/// With these two things in place, you should be able to use deserializeStruct
/// in QueryCache.
///
/// Currently the VM<->Orca interface (QueryCache and the callback machinery) 
/// supports structs passed by value only. If you ever need to pass a
/// dynamically allocated structure passed (e.g. you want to pass a dynamically 
/// sized array) look through the git history. We once had that for ObjectLayout
/// type. 
template <typename T> struct Cacheable {
  template <typename FieldTraversalTy>
  static T create(FieldTraversalTy &FT) {
    T F;
    F.template traverseFields<FieldTraversalTy>(FT);
    return F;
  }
};

/// Information pertaining to a single field of a source level object.
struct FieldInfo : public Cacheable<FieldInfo> {
  friend struct Cacheable<FieldInfo>;
  
  enum FieldType {
    FIELD_TYPE_UNKNOWN = 0,
    FIELD_TYPE_BOOLEAN,
    FIELD_TYPE_BYTE,
    FIELD_TYPE_SHORT,
    FIELD_TYPE_INT,
    FIELD_TYPE_LONG,
    FIELD_TYPE_FLOAT,
    FIELD_TYPE_DOUBLE,
    FIELD_TYPE_CHAR,
    FIELD_TYPE_REFERENCE,
  };
  
  std::string Name;
  int32_t OffsetInBytes;
  uint32_t SizeInBytes;
  uint32_t AlignmentInBytes;

  // Reference type field properties
  // TODO: dereferenceability can be derived from the type:
  //   getJavaTypeInfo(KlassID)->getObjectSize()
  uint32_t DereferenceableInBytes;
  azul::TypeUtils::CompileTimeKlassID KlassID;

  llvm::APInt KnownZero;
  llvm::APInt KnownOne;

  uint32_t Flags;
  uint32_t Type;

  FieldInfo() {}

  void verify() {
    assert(KnownZero.getBitWidth() == SizeInBytes * 8 &&
           "Sizes needs to be in sync!");
    assert(KnownOne.getBitWidth() == SizeInBytes * 8 &&
           "Sizes needs to be in sync!");
    assert((isReferenceField() || DereferenceableInBytes == 0) &&
           "DereferenceableInBytes must be set for reference fields only");
    assert((isReferenceField() ||
            KlassID == azul::TypeUtils::CompileTimeKlassID(0)) &&
           "KlassID must be set for reference fields only");
    assert((isReferenceField() || ((Flags & REFERENCE_ONLY_FLAGS) == 0)) &&
           "reference field flags must be used only with reference fields");
    assert(((Flags & FLAG_KNOWN_TYPE) != 0 ||
            (Flags & FLAG_KNOWN_EXACT_TYPE) == 0) &&
           "FLAG_KNOWN_TYPE must be set if FLAG_KNOWN_EXACT_TYPE is set");
  }

public:
  enum : uint32_t {
    FLAG_NONE = 0,

    /// Loads from this field don't alias with any other stores in the
    /// runtime.  Loads of this field can be marked with !invariant.load
    FLAG_INVARIANT_FIELD = 2,

    /// The type of this reference field is known and stored in KlassID
    FLAG_KNOWN_TYPE = 4,

    /// This reference field is known to never be null
    FLAG_KNOWN_NON_NULL = 8,

    /// The type of this reference field is known to be exact
    FLAG_KNOWN_EXACT_TYPE = 16,

    // These flags are only make sense for reference type fields
    REFERENCE_ONLY_FLAGS =
        FLAG_KNOWN_TYPE | FLAG_KNOWN_NON_NULL | FLAG_KNOWN_EXACT_TYPE
  };

  explicit FieldInfo(int32_t O, uint32_t S)
      : OffsetInBytes(O), SizeInBytes(S), AlignmentInBytes(1),
        DereferenceableInBytes(0), KlassID(0),
        KnownZero(llvm::APInt(S * 8, 0)), KnownOne(llvm::APInt(S * 8, 0)),
        Flags(FLAG_NONE), Type(FIELD_TYPE_UNKNOWN) {
    verify();
  }

  void setName(llvm::StringRef N) { Name = N.str(); }
  void setKnownZero(llvm::APInt KZ) {
    assert(KZ.getBitWidth() == SizeInBytes * 8 &&
           "Sizes needs to be in sync!");
    KnownZero = KZ;
  }
  void setKnownOne(llvm::APInt KO) {
    assert(KO.getBitWidth() == SizeInBytes * 8 &&
           "Sizes needs to be in sync!");
    KnownOne = KO;
  }
  void setInvariantField() { Flags |= FLAG_INVARIANT_FIELD; }
  void setFieldType(FieldType T) { Type = T; }
  void setKnownNonNull() {
    assert(isReferenceField() && "makes sense for reference fields only");
    Flags |= FLAG_KNOWN_NON_NULL;
  }
  void setDereferenceableBytes(uint32_t D) {
    assert(isReferenceField() && "makes sense for reference fields only");
    DereferenceableInBytes = D;
  }
  void setAlignmentInBytes(uint32_t A) {
    assert(isReferenceField() && "makes sense for reference fields only");
    AlignmentInBytes = A;
  }
  void setJavaType(azul::TypeUtils::CompileTimeKlassID K, bool IsExact) {
    assert(isReferenceField() && "makes sense for reference fields only");
    KlassID = K;
    Flags |= FLAG_KNOWN_TYPE;
    if (IsExact)
      Flags |= FLAG_KNOWN_EXACT_TYPE;
  }

  bool isReferenceField() const {
    return Type == FIELD_TYPE_REFERENCE;
  }

  bool isInvariantField() const {
    return (Flags & FLAG_INVARIANT_FIELD) != 0;
  }

  StringRef getName() const { return Name; }
  int32_t getOffsetInBytes() const { return OffsetInBytes; }
  uint32_t getSizeInBytes() const { return SizeInBytes; }

  uint32_t getDereferenceableBytes() const {
    assert(isReferenceField() && "makes sense for reference fields only");
    return DereferenceableInBytes;
  }
  uint32_t getAlignmentInBytes() const {
    assert(isReferenceField() && "makes sense for reference fields only");
    assert(
        (AlignmentInBytes && !(AlignmentInBytes & (AlignmentInBytes - 1))) &&
        "should be power of 2!");
    return AlignmentInBytes;
  }

  bool isKnownNonNull() const {
    assert(isReferenceField() && "makes sense for reference fields only");
    return (Flags & FLAG_KNOWN_NON_NULL) != 0;
  }

  std::optional<TypeUtils::JavaType> getJavaType() const {
    assert(isReferenceField() && "makes sense for reference fields only");
    if ((Flags & FLAG_KNOWN_TYPE) == 0)
      return std::nullopt;

    const bool IsExact = (Flags & FLAG_KNOWN_EXACT_TYPE) != 0;
    return TypeUtils::JavaType(KlassID, IsExact);
  }

  const llvm::APInt &getKnownZero() const { return KnownZero; }
  const llvm::APInt &getKnownOne() const { return KnownOne; }

  bool isKnownPositive() const { return getKnownZero().isNegative(); }

  FieldType getFieldType() {
    return (FieldType) Type;
  }
  
  template <typename FieldTraversalTy>
  void traverseFields(FieldTraversalTy &FT) {
    FT.beginInstance("FieldInfo");
    FT.doFieldString("Name", Name);
    FT.doFieldInt32("OffsetInBytes", OffsetInBytes);
    FT.doFieldUInt32("SizeInBytes", SizeInBytes);
    FT.doFieldUInt32("AlignmentInBytes", AlignmentInBytes);
    FT.doFieldUInt32("DereferenceableInBytes",
                              DereferenceableInBytes);
    uint32_t ID = KlassID.getID();
    FT.doFieldUInt32("KlassID", ID);
    KlassID = azul::TypeUtils::CompileTimeKlassID(ID);
    FT.doFieldAPInt("KnownZero", KnownZero);
    FT.doFieldAPInt("KnownOne", KnownOne);
    FT.doFieldUInt32("Flags", Flags);
    FT.doFieldFieldType("Type", Type);
    FT.endInstance();
  }

  bool operator==(const FieldInfo& Other) const {
    return Name == Other.Name &&
           OffsetInBytes == Other.OffsetInBytes &&
           SizeInBytes == Other.SizeInBytes &&
           AlignmentInBytes == Other.AlignmentInBytes &&
           DereferenceableInBytes == Other.DereferenceableInBytes &&
           KlassID == Other.KlassID &&
           KnownZero == Other.KnownZero  &&
           KnownOne == Other.KnownOne &&
           Flags == Other.Flags &&
           Type == Other.Type; 
  }

  static bool CompareByOffset(const FieldInfo &X, const FieldInfo &Y) {
    return X.getOffsetInBytes() < Y.getOffsetInBytes();
  }
};

class JavaTypeInfo : public Cacheable<JavaTypeInfo> {
  friend struct Cacheable<JavaTypeInfo>;

public:
  enum : uint32_t {
    FLAG_NONE = 0,
    FLAG_IS_INTERFACE = 1,
    FLAG_IS_ABSTRACT = 2,
    FLAG_IS_OBJECT_SIZE_EXACT = 4,
    FLAG_IS_ARRAY = 8,
    FLAG_IS_INSTANCE = 16,
  };
  
  JavaTypeInfo() = default;
  explicit JavaTypeInfo(std::string N, uint32_t OS, uint32_t AHS, uint32_t AES,
                        uint32_t F)
      : Name(N), ObjectSize(OS), ArrayHeaderSize(AHS), ArrayElementShift(AES),
        Flags(F) {
    assert((ArrayElementShift == 0 || isArray()) &&
           "ArrayElementShift must be 0 for non-arrays");
  }


  StringRef getName() const { return Name; }
  uint32_t getObjectSize() const { return ObjectSize; }
  uint32_t getArrayHeaderSize() const { 
    assert(isArray() && "Doesn't make sense for non-arrays!");
    return ArrayHeaderSize; 
  }
  uint32_t getArrayElementShift() const {
    assert(isArray() && "Doesn't make sense for non-arrays!");
    return ArrayElementShift;
  }

  bool isInterface() const {
    return (Flags & FLAG_IS_INTERFACE) != 0;
  }
  bool isAbstract() const {
    // For now when the VM doesn't know about FLAG_IS_ABSTRACT and only sets 
    // FLAG_IS_INTERFACE implicitly treat interface types as abstract.
    // TODO: Once the VM is taught to set FLAG_IS_ABSTRACT assert that interface
    // is also abstract in the constructor and remove this special case.
    if (isInterface())
      return true;
    return (Flags & FLAG_IS_ABSTRACT) != 0;
  }
  bool isObjectSizeExact() const {
    return (Flags & FLAG_IS_OBJECT_SIZE_EXACT) != 0;
  }

  /// Return true if the type is known to be an array type.
  bool isArray() const {
    return (Flags & FLAG_IS_ARRAY) != 0;
  }

  /// Return true if the type is known to be an instance (non-array) type.
  bool isInstance() const {
    return (Flags & FLAG_IS_INSTANCE) != 0;
  }
  
  template <typename FieldTraversalTy>
  void traverseFields(FieldTraversalTy &FT) {
    FT.beginInstance("JavaTypeInfo");
    FT.doFieldString("Name", Name);
    FT.doFieldUInt32("ObjectSize", ObjectSize);
    FT.doFieldUInt32("ArrayHeaderSize", ArrayHeaderSize);
    FT.doFieldUInt32("ArrayElementShift", ArrayElementShift);
    FT.doFieldUInt32("Flags", Flags);
    FT.endInstance();
  }
  
  bool operator==(const JavaTypeInfo& Other) const {
    return Name == Other.Name &&
           ObjectSize == Other.ObjectSize &&
           ArrayHeaderSize == Other.ArrayHeaderSize &&
           ArrayElementShift == Other.ArrayElementShift && 
           Flags == Other.Flags;
  }

private:
  std::string Name;
  uint32_t ObjectSize;
  uint32_t ArrayHeaderSize;    // Must be zero for non-arrays
  uint32_t ArrayElementShift;  // Must be zero for non-arrays
  uint32_t Flags;
};

class FunctionInfo : public Cacheable<FunctionInfo> {
  friend struct Cacheable<FunctionInfo>;

public:
  
  FunctionInfo() = default;
  explicit FunctionInfo(uint64_t MethodID) : MethodID(MethodID) {}

  uint64_t getMethodID() const { return MethodID; }
  
  template <typename FieldTraversalTy>
  void traverseFields(FieldTraversalTy &FT) {
    FT.beginInstance("FunctionInfo");
    FT.doFieldUInt64("MethodID", MethodID);
    FT.endInstance();
  }
  
  bool operator==(const FunctionInfo& Other) const {
    return MethodID == Other.MethodID;
  }

private:
  uint64_t MethodID;
};

/// This enum is used to describe known facts about call argument for
/// _get_constant_result_callback and _get_specialized_implementation_callback
/// callbacks. See the comment in azul::orca::Callbacks for details.
enum ArgumentInfo {
  ARGUMENT_INFO_NONE = 0,
  ARGUMENT_INFO_IS_CONSTANT = 1,
  ARGUMENT_INFO_HAS_KNOWN_TYPE = 2, // Can only be set for pointer type args 
  ARGUMENT_INFO_HAS_EXACT_TYPE = 4,  // Can only be set for pointer type args
  LLVM_MARK_AS_BITMASK_ENUM(/* LargestFlag = */ ARGUMENT_INFO_HAS_EXACT_TYPE)
};

/// Optimize the specified module by applying the pass ordering specified by the
/// combination of TargetMachine and Level.
void optimize(llvm::Module *M, llvm::TargetMachine *TM, OptimizationLevel Level,
              const OrcaFeatures &Features = OrcaFeatures::getEmpty(),
              llvm::raw_ostream *QueryCacheOut = nullptr,
              llvm::raw_ostream *TimePassesStream = nullptr,
              llvm::json::OStream *CompileInfoOut = nullptr,
              llvm::StringRef *QueryCacheIn = nullptr);

/// Iterate over all queries in the serialized query cache and try to replace
/// std::nullopt with a more definite answer using VM callbacks. If at least
/// one query modified, serialize back the cache. Return a number of updated
/// queries.
unsigned updateQueryCache(llvm::SmallVectorImpl<char>& SerializedQueryCache);

/// Serialize the query cache "associated with" \p Ctx if
/// serialization was enabled when optimization started.
///
/// A query cache is created and gets "associated with" a LLVMContext
/// when azul::orca::optimize is invoked on a llvm::Module on that
/// LLVMContext, and ceases be associated with the same LLVMContext
/// once the orca::azul::optimize call returns.
///
/// The query cache is serialized to the stream passed in to the
/// azul::orca::optimize call (there must be one if serialization is
/// enabled).
///
/// NB: This function can be called from a signal handler and data
/// structures may not be in a consistent state when this is called.
/// This routine need to be resilient and salvage as much as it can.
void serializeQueryCache(llvm::LLVMContext &Ctx,
                         llvm::raw_ostream *QueryCacheOut = nullptr);

/// Matches the serialized query cache passed as an argument with the current
/// state of the system.
///
/// Replays the cache by asking the corresponding VM callbacks the same
/// questions and matches the answers from the VM with the results from the
/// cache. If all the results match it returns true, otherwise returns false.
/// 
/// LogMismatchCallback argument is the callback responsible for the mismatch
/// reporting. It receives three StringRef's: query, vm result and cached result 
/// respectively. For example implementation may choose to dump these strings 
/// to file.
///
/// If ExactMatch is false conservative answers in the given query cache will
/// be considered "matching" with more specific answers from the VM. Otherwise 
/// the queries should match precisely for the query cache to match.
using LogMismatchCallbackType = 
    std::function<void (llvm::StringRef, llvm::StringRef, llvm::StringRef)>;
bool matchQueryCache(llvm::StringRef SerializedQueryCache,
                     LogMismatchCallbackType LogMismatchCallback,
                     bool ExactMatch = true);

/// Returns SHA of current commit of the git source history,
/// this Orca was built from. Returns empty string if unknown.
StringRef orca_internal_source_version();

} // end namespace orca
} // end namespace azul

namespace llvm {
class OrcaOpt : public PassInfoMixin<OrcaOpt> {
  TargetMachine *TM;

public:
  explicit OrcaOpt(TargetMachine *_TM) : TM(_TM) {}
  PreservedAnalyses run(Module &M, ModuleAnalysisManager &AM);

  static bool
  parseOrcaPipelines(StringRef Name, ModulePassManager &PM,
                     ArrayRef<PassBuilder::PipelineElement> InnerPipeline,
                     Module &M, PassBuilder &PB);
};

/// This pass may have practical uses, but it's main purpose is to make it
/// easier to test counters. It just increments a counter with the given ID.
class OrcaIncrementCounterPass
    : public PassInfoMixin<OrcaIncrementCounterPass> {
  std::string CounterID;

public:
  explicit OrcaIncrementCounterPass(
      StringRef CounterID = "OrcaIncrementCounterPass")
      : CounterID(CounterID) {}

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);
};
}

// These preprocessor defines let Orca clients version their code so that it
// works against different versions of Orca.  These can and should be removed
// after approximately two weeks.

// This macro was added Oct 12, 2022
#define ORCA_COMPAT_ZING_FAST_NEW_STUB_CALLING_CONV2

// This macro was added Oct 18, 2022
#define ORCA_COMPAT_COMPILETIMEKLASSID

// This macro was added Nov 8, 2022
#define ORCA_COMPAT_RTHREAD_STR

// This macro was added Nov 14, 2022
#define ORCA_COMPAT_MEM_ATTR

// This macro was added Nov 17, 2022
#define ORCA_COMPAT_MODREF_HEADER

// This macro was added Nov 28, 2022
#define ORCA_COMPAT_OPTPASSGATE

// This macro was added Dec 07, 2022
#define ORCA_COMPAT_USE_STD_OPTIONAL

// This macro was added Dec 19, 2022
#define ORCA_COMPAT_WAIT_OPT

// This macro was added Dec 20, 2022
#define ORCA_COMPAT_INSERT_INTO

// This macro was added Dec 20, 2022
#define ORCA_COMPAT_JSON_OPTIONAL

// This macro was added Dec 21, 2022
#define ORCA_COMPAT_VM_CALLBACKS_STD_OPTIONAL

// This macro was added Dec 26, 2022
#define ORCA_COMPAT_STRING_MAP_ENTRY_CREATE

// This macro was added Jan 5, 2022
#define ORCA_COMPAT_LOCKOPT_REORDER_OPTION

// This macro was added Jan 31, 2023
#define ORCA_COMPAT_O4_OPTLEVEL

// This macro was added July 6, 2023
#define ORCA_COMPAT_NEW_ARRAY_ADDED_ZERO_FROM

// This macro was added again Jan 22, 2023
#define ORCA_COMPAT_NOP_ALLOC_BARRIER

// This macro was added Mar 28, 2023
#define ORCA_COMPAT_OPINFO

// This macro was added July 21, 2023
#define ORCA_COMPAT_OPAQUE_DEPRECATED

// This macro was added July 3, 2023
#define ORCA_COMPAT_SUBTARGET_FEATURE_MOVED

// This macro was added September 18, 2023
#define ORCA_COMPAT_CODEGENOPTLEVEL

// This macro was added October 7, 2023
#define ORCA_COMPAT_CHANGE_VFABI_MANGLE_API_CHANGE

// This macro was added Dec 7, 2023
#define ORCA_COMPAT_CHANGE_TEXT_SECTION_FLAG

#endif
