//===-- Metadata.h ------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// Defines the constant strings for our custom metadata types
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_METADATA_H
#define LLVM_AZUL_METADATA_H
namespace azul {
  /// Terminology & When to Strip:
  /// - All metadata must be stripped when replacing operands.  Whether the
  /// metadata describes the operand or the result of the instruction, in
  /// general replacing one of the operands invalidates the facts.  An example
  /// of a problematic transform would be:
  /// select(c, load1, load2) -> load (select(c, addr1, addr2))
  /// - Facts about results of expressions must be stripped when adding new
  /// uses.  (The original facts may have only held for a given set of uses.)
  /// Examples of problematic transforms include: any form of CSE
  /// - Some metadata might be control dependent, that is the facts implied by
  /// the metadata may depend on the control flow required to reach the
  /// original location of the instruction.  Such metadata must be stripped
  /// when speculating the instruction into any dynamic path on which it did
  /// not originally run. In general, facts about the *result* of an
  /// instruction are general control dependent, but facts about *inputs* may
  /// or may not be.  Examples of problematic transforms would be: LICM, 
  /// phi-to-select conversion for diamond patterns, etc..
  /// ***************************
  /// IF YOU ARE NOT SURE, STRIP!
  /// ***************************
  /// Our healing mechanism will manage to restore most metadata which could
  /// have been legally preserved.  The loss from stripping is mostly in
  /// compile time.

  /// known_value indicates that the object reachable through a pointr so
  /// annotated is known to be a particular object.  We do not neccessarily
  /// known the *address* of that object and if we're still compiling against
  /// the abstract model of non-integral pointers before lowering, the pointer
  /// may infact be a reference whose bitpattern changes across time.  Note
  /// that known_value can used for both GC managed and non-GC managed objects
  /// and there is no restriction that only non-integral pointers/references be
  /// annotated. This fact is control dependent.  
  constexpr const char *MD_KnownValue = "known_value";
  
  /// azul.lvb_mask_load is used to tag the load of the LVB trap mask.  It is
  /// in practice a property of the address loaded from and thus is not control
  /// dependent.
  constexpr const char *MD_LVBMaskLoad = "azul.lvb_mask_load";

  /// azul-base-pointer-java-type indicates the JavaType of the object within
  /// which the annotated pointer must point.  It is only legal to use on
  /// pointers which are known to be based on a single object (allocation) at
  /// runtime.  "based-on" here is the same definition used for AA.  This fact
  /// is control-dependent and use-dependent.
  /// Note that azul-base-pointer-java-type contains a compile time klass ID.
  constexpr const char *MD_BasePointerJavaType = "azul-base-pointer-java-type";

  constexpr const char *MD_JavaTypeKid = "java-type-kid";
  constexpr const char *MD_JavaTypeExact = "java-type-exact";

  /// Loads tagged with this metadata need not be LVB'd. It is a property of the
  /// address loaded from and thus is not control dependent.
  constexpr const char *MD_AvoidLVBOnLoad = "azul.avoid_lvb_on_load";

  /// The next two metadata is tagged on safepoint_polls which is used during our
  /// optimized safepoint polls pass to choose a loop for speculation or
  /// chunking.

  /// azul.speculatable_safepoint is used to tag safepoints as speculatable, and
  /// when all safepoint polls within a loop are tagged with this metadata, an
  /// uncounted loop can be speculated as finite for optimizing safepoint polls.
  /// (which is added by the frontend when generating IR) to avoid repeated
  /// deoptimization based on this speculation.
  constexpr const char *MD_SpeculatableSafepoint =
      "azul.speculatable_safepoint";

  /// azul.not_speculatable_safepoint is tagged on safepoints when the uncounted
  /// loop speculation failed. This is to identify if uncounted loop speculation
  /// has failed in which case we fallback to loop chunking during OSP.
  /// MD_SpeculatableSafepoint and MD_UncountedLoopSpeculationFailed are mutually
  /// exclusive.
  constexpr const char *MD_UncountedLoopSpeculationFailed =
      "azul.not_speculatable_safepoint";

  /// A loop tagged w/this loop metatdata will not have value unswitching
  /// performed on it.  (i.e. for hybrid lite)
  constexpr const char* MD_DisableValueSpeculation =
    "azul.disable_value_speculation";

  constexpr const char* MD_NumInlinedIntoTopLevel =
    "azul.num.inlined.into.top.level";

  constexpr const char* MD_ExtraOptArgs = "azul.extra.opt.args";

  constexpr const char* MD_KnownDeoptBundleFormat =
    "azul.known.deopt.bundle.format";

  /// A module level metadata that stores the Java types of new instance
  /// allocations that exist in the top-level function. It is used to remember
  /// the types before we lower or otherwise rewrite new allocation abstraction
  /// calls.
  ///
  /// OrcaAA relies on this metadata to figure out the types of new instance
  /// allocations after allocation abstraction lowering.
  ///
  /// If the metadata has no operands it indicates that the types of new
  /// allocations are unknown. Otherwise the metadata must have one operand
  /// pointing to a JavaType union metadata node:
  ///
  ///   !azul.new.instance.types = !{!0}
  ///
  ///   !0 = !{!0, ..., !n}
  ///   !1 = !{i64 <KlassID>, i1 <IsExact>}
  ///   ...
  ///   !n = !{i64 <KlassID>, i1 <IsExact>}
  constexpr const char* MD_NewInstanceTypes = "azul.new.instance.types";

  /// A module level metadata indicating that the module has explicit 
  /// relocation. This metadata is set by RS4GC.
  constexpr const char* MD_HasExplicitRelocations =
    "azul.has.explicit.relocations";

  /// A module level metadata indicating that the module has compressed 
  /// oops. This metadata is set by VM.
  constexpr const char* MD_HasCompressedOops = "azul.has.compressed.oops";

  /// A module level metadata indicating that the toplevel function in the
  /// module needs LVBs. This is set through the VM. 
  constexpr const char* MD_AddLVBs = "azul.add.lvbs";

  /// A module level metadata indicating that the toplevel function in the
  /// module needs SVBs. This is set through the VM. 
  constexpr const char* MD_AddSVBs = "azul.add.svbs";

  /// azul.droppable_safepoint is used to tag safepoints that can be dropped
  /// by OSP. This is used to identify safepoints that are required for
  /// correctness of some transformations performed by OSP (counted loop
  /// speculation might, for instance, require a poll in loop predecessor). If
  /// the i1 value is 0, OSP shouldn't remove the poll. If a poll doesn't have
  /// this metadata, there's a transformation that doesn't preserve it and we
  /// have to fix it.
  constexpr const char *MD_DroppableSafepoint = "azul.droppable_safepoint";

  /// azul.speculatable_loop_nest tag on a safepoint means that loop nest
  /// speculation never failed on this specific safepoint. Loop nest speculation
  /// should be applied only to loop nest in which all safepoints have this tag.
  constexpr const char *MD_SpeculatableLoopNest = "azul.speculatable_loop_nest";

  /// Identify pdep and pext instructions that are part of the
  /// compress/uncompress routine. This is used for optimizing patterns in
  /// instCombine and for quickly checking if compressedOops is in the module.
  constexpr const char *MD_CompressedOops = "azul.compressed_oops";

  /// Indicates that the value is a cached box object.
  /// A box object is a wrapper over a primitive value produced as a
  /// result of a call to the corresponding valueOf method.
  /// A cached box object means that:
  ///   BoxType.valueOf(value.boxTypeValue) == value.
  constexpr const char *MD_CachedBoxObject = "azul.cached_box_object";

  // When an LVB has this metadata, it has only explicit uses seen in the IR.
  // All LVBs originally added as part of AddGCBarriers will have this
  // metadata. LVB optimizations can only be applied on LVB that has this
  // metadata. For example, if an LVB (on a load L) is only used in a null check
  // but is missing this metadata, it means the LVB is implicitly needed for
  // another load (L') from the same address and hence we cannot apply the LVB
  // elimination. The metadata is stripped when performing optimizations that
  // reuse LVBs for multiple loads from same address. 
  constexpr const char *MD_LVBHasNoImpliedUse = "azul.lvb_has_no_implied_use";

  // An instruction marker for print-uses pass.
  constexpr const char *MD_InstructionMarker  = "azul.instruction_marker";
} // namespace azul

#endif
