//===- JVMStateBundle.h - Helpers for deopt bundles for the JVM -*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef LLVM_IR_JVM_STATE_BUNDLE_H
#define LLVM_IR_JVM_STATE_BUNDLE_H

#include "llvm/ADT/ArrayRef.h"
#include "llvm/ADT/BitmaskEnum.h"
#include "llvm/ADT/DenseMap.h"

#include "llvm/IR/Constants.h"
#include "llvm/IR/InstrTypes.h"
#include "llvm/IR/ValueHandle.h"
#include "llvm/IR/ValueMap.h"

#include <map>
#include <optional>

namespace llvm {
bool isKnownDeoptBundleFormat(const CallBase *CB);
void writeBriefDeoptBundleAnnotation(const Module *M, OperandBundleUse BU,
                                     raw_ostream &Out, StringRef Prefix = "");

class AbstractFrameBase {
public:
  enum ValueKind : unsigned {
    FirstValueKind = 0,
    StackValue = FirstValueKind,
    LocalValue,
    Monitor,
    NumValueKinds
  };

protected:
  enum : unsigned {
    FlagIndex = 0,
    CallerIDIndex,
    BCIIndex,
    NumStackElementsIndex,
    NumLocalsIndex,
    NumMonitorIndex,
    HeaderSize
  };
};

/// \brief This class specifies layout of elements of a single abstract frame.
///
/// To use this class you have to derive your class from it and pass your class
/// as a template parameter. Your class should implement following methods:
/// getNumStackElements, getNumLocals and getNumMonitors.
template <class T>
class AbstractFrameLayout : public AbstractFrameBase {
protected:
  /// Returns an offset at which the first stack element should be.
  unsigned getStackElementsBeginOffset() const { return HeaderSize; }
  /// Returns an offset which is the next to the end of the last stack element.
  unsigned getStackElementsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getStackElementsBeginOffset() + ThisT->getNumStackElements() * 2;
  }

  /// Returns an offset at which the first local value should be.
  unsigned getLocalsBeginOffset() const { return getStackElementsEndOffset(); }
  /// Returns an offset which is the next to the end of the last local value.
  unsigned getLocalsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getLocalsBeginOffset() + ThisT->getNumLocals() * 2;
  }

  /// Returns an offset at which the first monitor should be.
  unsigned getMonitorsBeginOffset() const { return getLocalsEndOffset(); }
  /// Returns an offset which is the next to the end of the last monitor.
  unsigned getMonitorsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getMonitorsBeginOffset() + ThisT->getNumMonitors();
  }

  unsigned getValuesBeginOffset(ValueKind Kind) const {
    switch (Kind) {
    case StackValue:
      return getStackElementsBeginOffset();
    case LocalValue:
      return getLocalsBeginOffset();
    case Monitor:
      return getMonitorsBeginOffset();
    default:
      llvm_unreachable("Invalid value kind!");
    }
  }

  unsigned getValuesEndOffset(ValueKind Kind) const {
    switch (Kind) {
    case StackValue:
      return getStackElementsEndOffset();
    case LocalValue:
      return getLocalsEndOffset();
    case Monitor:
      return getMonitorsEndOffset();
    default:
      llvm_unreachable("Invalid value kind!");
    }
  }

public:
  /// Calculates length of an abstract frame. Length here is number of Values
  /// required to encode state of a frame.
  unsigned getLength() const {
    auto ThisT = static_cast<const T *>(this);
    return HeaderSize + ThisT->getNumStackElements() * 2 +
           ThisT->getNumLocals() * 2 + ThisT->getNumMonitors() * 2;
  }
};

/// \brief Represents a single abstract inlined frame in a full deoptimization
/// state.
class AbstractFrame : public AbstractFrameLayout<AbstractFrame> {
public:
  /// Specifies possible flags for Abstract frame.
  enum Flags : unsigned {
    FLAG_NONE = 0,

    // Frame state corresponds to bytecode before execution if 0 or after if 1.
    FLAG_REEXECUTABLE = 1 << 0,

    // The flag indicates that if exception happens then we should deoptimize.
    FLAG_DEOPT_ON_THROW = 1 << 1,

    // Frame state contains extra lock. We should release it if we deoptimize.
    FLAG_EXTRA_LOCK = 1 << 2,

    // There is an OOM handler for this location in the current frame that
    // can access local references.
    FLAG_OOM_HANDLER_INSPECTS_LOCALS = 1 << 3,

    // Frame state contains an eliminated nested lock (which is pushed as the last lock on
    // the stack through PushMonitor, similar to as done for FLAG_EXTRA_LOCK).
    // If we deoptimize we should lock it and drop it from the monitor stack.
    FLAG_ELIMINATED_LOCK = 1 << 4,

    LLVM_MARK_AS_BITMASK_ENUM(
        /* LargestValue = */ FLAG_ELIMINATED_LOCK)
  };

private:
  Flags Flag;
  uint32_t CallerID;
  uint32_t BCI;
  uint32_t NumStackElements;
  uint32_t NumLocals;
  uint32_t NumMonitors;

  ArrayRef<Use> AbstractState;

public:
  enum class ValueType : uint32_t {
    FirstType = 0,
    OopValue = FirstType,
    FirstPrimitiveType,
    Primitive8 = FirstPrimitiveType,
    Primitive16,
    Primitive32,
    Primitive64,
    FirstNonPrimitiveType,
    Address = FirstNonPrimitiveType,
    LazyObject,
    Dead,
    KnownValue,
    NumValueTypes,
  };

  struct StateValue {
    ValueType Ty;
    Value *Val;

    StateValue(ValueType Ty, Value *Val) : Ty(Ty), Val(Val) {};
    
    static StateValue fromValues(Value *Ty, Value *Val);
    static bool isStateValue(Value *Ty, Value *Val);
    
    static bool canEncode(Type *Ty) {
      return getValueType(Ty).has_value();
    }
    static bool canEncode(Value *V) {
      return canEncode(V->getType());
    }
    
    bool isDead() const {
      return Ty == ValueType::Dead;
    }
    std::optional<Value *> asValue() const {
      if (Ty == ValueType::Dead || 
          Ty == ValueType::LazyObject ||
          Ty == ValueType::KnownValue)
        return std::nullopt;
      return Val;
    }

    bool operator==(const StateValue &SV) const {
      return Ty == SV.Ty && Val == SV.Val;
    };

    std::optional<uint32_t> asLazyObjectID() const {
      if (Ty != ValueType::LazyObject)
        return std::nullopt;
      return cast<ConstantInt>(Val)->getZExtValue();
    }

    std::optional<uint32_t> asKnownValueID() const {
      if (Ty != ValueType::KnownValue)
        return std::nullopt;
      return cast<ConstantInt>(Val)->getZExtValue();
    }

    static std::optional<StateValue> fromValue(Value *V) {
      if (auto VTY = getValueType(V->getType()))
        return StateValue(*VTY, V);
      return std::nullopt;
    }

    static StateValue fromKnownValue(LLVMContext &C, uint32_t KVID);
    static StateValue fromLazyObjectID(LLVMContext &C, uint32_t LOID);

    static std::optional<ValueType> getValueType(Type *Ty);
  };

private:
  AbstractFrame() = default;

public:
  /// \brief Construct an AbstractFrame instance from a sequence of operands.
  ///
  /// This parses out a single abstract frame from the *prefix* of \p FullState.
  /// The exact number of elements of \p FullState that are part of this
  /// abstract frame is reported by getLength.
  static std::optional<AbstractFrame> TryBuild(ArrayRef<Use> FullState);
  static AbstractFrame Build(ArrayRef<Use> FullState);

  bool hasAnyFlags() const { return Flag; }
  bool isReexecutable() const { return Flag & FLAG_REEXECUTABLE; }
  bool isDeoptOnThrow() const { return Flag & FLAG_DEOPT_ON_THROW; }
  bool hasExtraLock() const { return Flag & FLAG_EXTRA_LOCK; }
  bool hasEliminatedLock() const { return Flag & FLAG_ELIMINATED_LOCK; }
  bool canOOMHandlerInspectLocals() const {
    return Flag & FLAG_OOM_HANDLER_INSPECTS_LOCALS;
  }
  Flags getFlags() const { return Flag; }
  uint32_t getCallerID() const { return CallerID; }
  uint32_t getBCI() const { return BCI; }
  unsigned getNumStackElements() const { return NumStackElements; }
  unsigned getNumLocals() const { return NumLocals; }
  unsigned getNumMonitors() const { return NumMonitors; }

  // Returns number of values of the specified kind in this frame.
  unsigned getNumValues(ValueKind Kind) const;

  StateValue getValueAt(ValueKind Kind, unsigned Index) const;

  LLVMContext &getContext() const {
    return AbstractState.front()->getContext();
  }

  /// Returns AbstractState converted into vector of Value*.
  /// MutableAbstractJVMState constructs MutableAbstractFrames lazily, so this
  /// method is required to avoid constructing missing mutable frames when
  /// MutableAbstractJVMState::serialize is called.
  void serialize(std::vector<Value *> &Out) const;

  static StringRef ValueKindAsString(ValueKind Kind);

private:
  Value *getOperandAt(unsigned Index) const { return AbstractState[Index]; }
  StateValue getStateValueAtOffset(unsigned Offset) const;
};

/// Length, ID, KlassID, HasArrayLength, ArrayLength, LockCount
constexpr unsigned LAZY_OBJECT_HEADER_LENGTH = 6;
constexpr unsigned LAZY_OBJECT_FIELD_LENGTH = 4;

enum LazyObjectFlags {
  LAZY_OBJECT_FLAG_NONE = 0,
  LAZY_OBJECT_FLAG_ARRAY = 1,
  LAZY_OBJECT_FLAG_LAZY_BOX = 2
};

struct InitCommand {
  enum {
    // Command IDs.
    // Use small negative numbers to easily find them in IR dumps.
    FIELD = -1,  // Offset, StateValue(Value)
    MEMCPY = -2, // DstOffset, StateValue(Src), SrcOffset, Length, ElemSize

    // Number of arguments.
    NUM_OF_ARGS_FIELD = 3,
    NUM_OF_ARGS_MEMCPY = 6,
  };

  ArrayRef<Use> Args; // Includes command type as Args[0].

  InitCommand(ArrayRef<Use> Args) : Args(Args) {
    assert(getLength() == 1 + getNumOfArgs(getCmd()));
  }

  const Use *getCmdUse() const { return &Args[0]; };
  int32_t getCmd() const;
  unsigned getLength() const { return Args.size(); }

  static size_t isInitCommand(int InitCommand) {
    switch (InitCommand) {
    case FIELD:
    case MEMCPY:
      return true;
    }
    return false;
  }

  static size_t getNumOfArgs(int InitCommand) {
    switch (InitCommand) {
    case FIELD:
      return NUM_OF_ARGS_FIELD;
    case MEMCPY:
      return NUM_OF_ARGS_MEMCPY;
    }
    llvm_unreachable("Unknown init command");
  }

  template <typename VisitFunc>
  void visitStateValues(const VisitFunc &Visit) const;

  AbstractFrame::StateValue getMemcpySrc() const {
    assert(getCmd() == MEMCPY);
    return AbstractFrame::StateValue::fromValues(Args[2].get(), Args[3].get());
  }
};

class LazyObject {
  using StateValue = AbstractFrame::StateValue;

  uint32_t ID;
  Value *KlassID;
  std::optional<Value *> ArrayLength;
  unsigned LockCount = 0;
  /// Indicates that the lazy object is a lazily materialized box object.
  /// A box object is a wrapper over a primitive value produced as a
  /// result of a call to the corresponding BoxType::valueOf method.
  /// Some box types have a cache of small values which is required by the JLS.
  /// If the boxed value is within the cached range valueOf returns the cached 
  /// box, otherwise a new instance is allocated. Lazy materialization of boxes
  /// implements the same behavior.
  bool LazyBox = false;
  SmallVector<InitCommand, 8> InitCommands;

private:
  LazyObject() = default;

public:
  static std::optional<LazyObject> TryBuild(ArrayRef<Use> Uses);
  static LazyObject Build(ArrayRef<Use> Uses) { return *TryBuild(Uses); }

  uint32_t getID() const { return ID; }
  Value *getKlassID() const { return KlassID; }
  std::optional<Value *> getArrayLength() const { return ArrayLength; }
  unsigned getLockCount() const { return LockCount; }
  bool isLazyBox() const { return LazyBox; }

  unsigned getLength() const {
    unsigned Result = LAZY_OBJECT_HEADER_LENGTH;
    for (auto &IC : InitCommands)
      Result += IC.getLength();
    return Result;
  }

  const SmallVectorImpl<InitCommand> &getInitCommands() const {
    return InitCommands;
  }
};

class LazyObjectSection {
  ArrayRef<Use> FullSection;
  SmallVector<LazyObject, 8> LazyObjects;

private:
  LazyObjectSection() = default;

public:
  static std::optional<LazyObjectSection> TryBuild(ArrayRef<Use> Uses);
  static LazyObjectSection Build(ArrayRef<Use> Uses) { return *TryBuild(Uses); }
  static LazyObjectSection BuildEmpty() { return LazyObjectSection(); }

  unsigned getLength() const { return FullSection.size(); }
  unsigned getNumObjects() const { return LazyObjects.size(); }

  const LazyObject &getObjectAt(unsigned Index) const {
    return LazyObjects[Index];
  }

  void serialize(std::vector<Value *> &Out) const {
    Out.insert(Out.end(), FullSection.begin(), FullSection.end());
  }
};

/// \brief Represents a "full" deoptimization state, consisting of a sequence of
/// abstract frames.
///
/// Note: while this a wrapper class, constructing this isn't as cheap as
/// constructing a CallSite or an ArrayRef.
class AbstractJVMState {
  ArrayRef<Use> FullState;
  LazyObjectSection LazyObjects;
  SmallVector<AbstractFrame, 8> Frames;

private:
  AbstractJVMState() : LazyObjects(LazyObjectSection::BuildEmpty()) {};

public:
  AbstractJVMState(const AbstractJVMState &JVMS)
      : LazyObjects(LazyObjectSection::BuildEmpty()) {
    // Cannot delete this method because it is needed by
    // Optional<AbstractJVMState>.
    llvm_unreachable("Avoid copying AbstractJVMState");
  };
  AbstractJVMState &operator=(const AbstractJVMState &JVMS) {
    // Cannot delete this method because it is needed by
    // Optional<AbstractJVMState>.
    llvm_unreachable("Avoid copying AbstractJVMState");
  };

  AbstractJVMState(AbstractJVMState &&JVMS) = default;
  AbstractJVMState &operator=(AbstractJVMState &&JVMS) = default;

  static std::optional<AbstractJVMState> TryBuild(OperandBundleUse OBU);
  static AbstractJVMState Build(OperandBundleUse OBU);
  static AbstractJVMState BuildEmpty() { return AbstractJVMState(); }

  unsigned getNumFrames() const { return Frames.size(); }
  const AbstractFrame &getFrameAt(unsigned Index) const {
    return Frames[Index];
  }
  const AbstractFrame &getYoungestFrame() const {
    return Frames.back();
  }
  const LazyObjectSection &getLazyObjectSection() const {
    return LazyObjects;
  }
  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order.
  void visitStateValues(
      std::function<void(const AbstractFrame::StateValue &SV)> Visit) const;
};

/// \brief Represents a single abstract inlined frame in a full deoptimization
/// state. This class enables you to alter abstract frame contents and generate
/// a new state that contains your modifications.
///
/// Note: it doesn't affect exsisting state, it just remembers all
/// modifications you want to perform and generates NEW state that you can use
/// when you generate new deopt bundles.
class MutableAbstractFrame : public AbstractFrameLayout<MutableAbstractFrame> {
  using Flags = AbstractFrame::Flags;
  using ValueType = AbstractFrame::ValueType;

public:
  using StateValue = AbstractFrame::StateValue;

private:
  Flags Flag;
  uint32_t CallerID;
  uint32_t BCI;

  SmallVector<StateValue, 8> State[ValueKind::NumValueKinds];

public:
  MutableAbstractFrame(int function_id, int bci, Flags flags,
                       unsigned StackSize, unsigned LocalsSize,
                       unsigned MonitorsSize)
      : Flag(flags), CallerID(function_id), BCI(bci) {
    reserve(StackValue, StackSize);
    reserve(LocalValue, LocalsSize);
    reserve(Monitor, MonitorsSize);
  }

  explicit MutableAbstractFrame(int function_id, int bci, Flags flags)
    : Flag(flags), CallerID(function_id), BCI(bci) { }

  explicit MutableAbstractFrame(const AbstractFrame &Frame);

  bool hasAnyFlags() const { return Flag; }
  bool isReexecutable() const { return Flag & Flags::FLAG_REEXECUTABLE; }
  bool isDeoptOnThrow() const { return Flag & Flags::FLAG_DEOPT_ON_THROW; }
  bool hasExtraLock() const { return Flag & Flags::FLAG_EXTRA_LOCK; }
  bool hasEliminatedLock() const { return Flag & Flags::FLAG_ELIMINATED_LOCK; }
  bool canOOMHandlerInspectLocals() const {
    return Flag & Flags::FLAG_OOM_HANDLER_INSPECTS_LOCALS;
  }
  Flags getFlags() const { return Flag; }
  uint32_t getCallerID() const { return CallerID; }
  uint32_t getBCI() const { return BCI; }
  unsigned getNumStackElements() const { return getNumValues(StackValue); }
  unsigned getNumLocals() const { return getNumValues(LocalValue); }
  unsigned getNumMonitors() const { return getNumValues(Monitor); }
  void reserveStackElements(unsigned Size) { reserve(StackValue, Size); }
  void reserveLocals(unsigned Size) { reserve(LocalValue, Size); }
  void reserveMonitors(unsigned Size) { reserve(Monitor, Size); }

  void setFlags(Flags NewValue) { Flag = NewValue; }
  void addFlags(Flags NewValue) { Flag |= NewValue; }
  void setCallerID(uint32_t NewValue) { CallerID = NewValue; }
  void setBCI(uint32_t NewValue) { BCI = NewValue; }

  // Inserts a value of a given Kind at position specified by Index.
  // When position is at the end of corresponding vector:
  //   Index == getNumValues(Kind)
  // it appends the value to the end of the vector.
  void addStateValue(ValueKind Kind, unsigned Index, StateValue NewValue);

  void pushMonitor(Value *NewMonitor) {
    pushMonitor({ValueType::OopValue, NewMonitor});
  }

  void pushMonitor(StateValue NewMonitor) {
    addStateValue(ValueKind::Monitor, getNumMonitors(), NewMonitor);
  }

  void pushStackValue(StateValue NewStackValue) {
    addStateValue(ValueKind::StackValue, getNumStackElements(), NewStackValue);
  }

  void pushLocalValue(StateValue NewLocalValue) {
    addStateValue(ValueKind::LocalValue, getNumLocals(), NewLocalValue);
  }

  // Returns number of values of the specified kind in this frame.
  unsigned getNumValues(ValueKind Kind) const;

  // Reserves space for given amount of values of the specified kind.
  void reserve(ValueKind Kind, unsigned Size);

  StateValue &valueAt(ValueKind Kind, unsigned Index);

  void serialize(std::vector<Value *> &Out, LLVMContext &C) const;

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool
  changeStateValues(std::function<bool(AbstractFrame::StateValue &SV)> Change);
};

struct MutableInitCommand {
  int32_t Cmd;
  SmallVector<Value *, 8> Args;

  MutableInitCommand(int32_t Cmd) : Cmd(Cmd) {}
  MutableInitCommand(const InitCommand &IC);

  unsigned getLength() const { return 1 + InitCommand::getNumOfArgs(Cmd); }

  template <typename ChangeFunc>
  bool changeStateValues(const ChangeFunc &Change);
};

class MutableLazyObject {
  uint32_t ID;
  Value *KlassID;
  std::optional<Value *> ArrayLength;
  unsigned LockCount = 0;
  bool LazyBox = false;
  SmallVector<MutableInitCommand, 8> InitCommands;

public:
  explicit MutableLazyObject(const LazyObject &L)
      : ID(L.getID()), KlassID(L.getKlassID()), ArrayLength(L.getArrayLength()),
        LockCount(L.getLockCount()), LazyBox(L.isLazyBox()),
        InitCommands(parse(L.getInitCommands())) {}

  explicit MutableLazyObject(uint32_t ID, Value *KlassID,
                             std::optional<Value *> ArrayLength,
                             unsigned LockCount, bool LazyBox)
      : ID(ID), KlassID(KlassID), ArrayLength(ArrayLength),
        LockCount(LockCount), LazyBox(LazyBox) {
    assert((!LazyBox || LockCount == 0) &&
           "LazyBoxes can't have non-zero LockCount");
  }

  static SmallVector<MutableInitCommand, 8>
  parse(const SmallVectorImpl<InitCommand> &InitCommands);

  uint32_t getID() const { return ID; }
  Value *getKlassID() { return KlassID; }
  std::optional<Value *> getArrayLength() { return ArrayLength; }
  unsigned getLockCount() const { return LockCount; }
  bool isLazyBox() { return LazyBox; }

  void setKlassID(Value *KlassID) { this->KlassID = KlassID; }
  void setArrayLength(std::optional<Value *> ArrayLength) {
    this->ArrayLength = ArrayLength;
  }
  void setLockCount(unsigned LockCount) {
    this->LockCount = LockCount;
  }
  void setLazyBox() { LazyBox = true; }

  SmallVectorImpl<MutableInitCommand> &initCommands() { return InitCommands; }
  const SmallVectorImpl<MutableInitCommand> &initCommands() const {
    return InitCommands;
  }

  unsigned getLength() const {
    unsigned Length = LAZY_OBJECT_HEADER_LENGTH;
    for (auto &IC : InitCommands)
      Length += IC.getLength();
    return Length;
  }

  void serialize(std::vector<Value *> &Out, LLVMContext &Context) const;

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool
  changeStateValues(std::function<bool(AbstractFrame::StateValue &SV)> Change);
};

class MutableLazyObjectSection {
  // TODO: Make MutableLazyObjectSection lazy initialized with the immutable
  // LazyObjectSection and mutable creating objects if a change is made. That is
  // similar to the mutable frames in MutableAbstractJVMState.
  // std::map is used here to get the natural order from the iterator, so the
  // lazy objects get into the lazy section in order of their numbers.
  using LazyObjectMap = std::map<uint32_t, MutableLazyObject>;
  LazyObjectMap LazyObjects;
  unsigned MaxID;

public:
  explicit MutableLazyObjectSection(const LazyObjectSection &L) : MaxID(0) {
    for (unsigned Index = 0; Index < L.getNumObjects(); ++Index) {
      auto Object = L.getObjectAt(Index);
      LazyObjects.emplace(Object.getID(), MutableLazyObject(Object));
      MaxID = std::max(MaxID, Object.getID());
    }
  }

  void serialize(std::vector<Value *> &Out, LLVMContext &C) const;

  MutableLazyObject &addLazyObject(Value *KlassID,
                                   std::optional<Value *> ArrayLength,
                                   unsigned LockCount, bool LazyBox) {
    auto NewID = ++MaxID;
    auto O = LazyObjects.emplace(NewID,
      MutableLazyObject(NewID, KlassID, ArrayLength, LockCount, LazyBox));
    return O.first->second;
  }

  LazyObjectMap::iterator begin() { return LazyObjects.begin(); }
  LazyObjectMap::iterator end() { return LazyObjects.end(); }
};

/// \brief Represents a "full" deoptimization state, consisting of a sequence of
/// abstract frames. You can modify contents of abstract frames and generate new
/// deoptimization state that contains your modifications.
///
/// Note: it doesn't affect any existing "deopt" bundle! To use your new state
/// you have to generate a new instruction with this state in its deopt bundle.
class MutableAbstractJVMState {
  AbstractJVMState JVMS;
  DenseMap<unsigned, MutableAbstractFrame> Frames;
  int NumFrames;
  MutableLazyObjectSection Objects;

  // This map contains references to allocation instructions added by
  // addDematerializableAllocation(Instruction*). All these allocations are
  // prepared for being dematerialized by dematerializeLazyObjects().
  DenseMap<Value *, unsigned> NewLazyObjects;

public:
  explicit MutableAbstractJVMState(AbstractJVMState &&JVMS)
      : JVMS(std::move(JVMS)), NumFrames(this->JVMS.getNumFrames()),
        Objects(this->JVMS.getLazyObjectSection()) {}
  // Constructs AbstractJVMState and then uses it to construct
  // MutableAbstractJVMState. Fails only when AbstractJVMState::TryBuild fails.
  static std::optional<MutableAbstractJVMState> TryBuild(OperandBundleUse OBU);
  static MutableAbstractJVMState Build(OperandBundleUse OBU);

  static MutableAbstractJVMState BuildEmpty() {
    return MutableAbstractJVMState{AbstractJVMState::BuildEmpty()};
  }

  unsigned getNumFrames() const { return NumFrames; }
  // Returns a reference to mutable abstract frame that you can modify. Use
  // method `serialize` to get abstract state that contains your modifications.
  MutableAbstractFrame &getFrameAt(unsigned Index);
  MutableAbstractFrame &getYoungestFrame() {
    return getFrameAt(getNumFrames() - 1);
  }
  MutableLazyObjectSection &getLazyObjectSection() { return Objects; }

  MutableAbstractFrame &appendFrame(const AbstractFrame &F);
  MutableAbstractFrame &appendFrame(MutableAbstractFrame &&F);

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool
  changeStateValues(std::function<bool(AbstractFrame::StateValue &SV)> Change);

  // Adds a new lazy object without a need for later allocation references
  // fixing, as opposed to addDematerializableAllocation(Instruction*).
  MutableLazyObject &addLazyObject(Value *KlassID,
                                   std::optional<Value *> ArrayLength,
                                   unsigned LockCount, bool LazyBox) {
    return Objects.addLazyObject(KlassID, ArrayLength, LockCount, LazyBox);
  }

  // Adds a new lazy object for the specified Allocation. References to all
  // allocations must be dematerialized by method
  // dematerializeLazyObjects(LLVMContext&) before this MutableAbstractJVMState
  // is serialized.
  MutableLazyObject &addDematerializableObject(Instruction *Alloc);
  MutableLazyObject &
  addDematerializableObject(Value *KlassID, std::optional<Value *> ArrayLength,
                            unsigned LockCount, bool LazyBox,
                            const SmallVectorImpl<Value *> &Aliases);
  MutableLazyObject &
  addDematerializableObject(Value *KlassID, std::optional<Value *> ArrayLength,
                            unsigned LockCount, bool LazyBox, Value *V) {
    return addDematerializableObject(KlassID, ArrayLength, LockCount,
                                     LazyBox, SmallVector<Value *, 1>(1, V));
  }

  // Once both adding lazy objects and adding their fields are finished
  // this method should be called before serialization can be done.
  // This method changes all references StateValue(OopType, Value*) to the
  // lazy object references StateValue(LazyObject, LazyObjectID).
  // Returns true if at least one change has been made.
  void dematerializeLazyObjects(LLVMContext &C);

  // Returns new abstract deoptimization state that contains all the
  // modifications performed on states returned by getFrameAt/getYoungestFrame.
  std::vector<Value *> serialize(LLVMContext &C) const;

private:
  void serializeFrameAt(std::vector<Value *> &Out, LLVMContext &C,
                        unsigned FrameIndex) const;
};

std::vector<Value *> ComposeDeoptBundles(Instruction *I,
                                         OperandBundleUse &ParentBundle,
                                         OperandBundleUse &ChildBundle,
                                         Instruction *ParentCall);

struct LazyObjectBuilder {
  struct InitCommandVH {
    int32_t Cmd;
    SmallVector<TrackingVH<llvm::Value>, 8> Args;

    InitCommandVH(int32_t Cmd) : Cmd(Cmd) {}
    static std::optional<InitCommandVH> GetField(Value *Offset, Value *Value);
    static std::optional<InitCommandVH> GetMemcpy(Value *DstOffset, Value *Src,
                                                  Value *SrcOffset,
                                                  Value *Length,
                                                  Value *ElemSize);
  };

  LazyObjectBuilder() = default;
  LazyObjectBuilder(Instruction *Alloc);
  LazyObjectBuilder(Value *KlassID, std::optional<Value *> ArrayLength);

  Value *KlassID;
  std::optional<TrackingVH<llvm::Value>> ArrayLength;
  unsigned LockCount = 0;
  bool LazyBox = false;

  SmallVector<InitCommandVH, 4> InitCommands;

  std::optional<llvm::Value *> getArrayLength() const {
    if (ArrayLength.has_value())
      return ArrayLength->getValPtr();
    return std::nullopt;
  }
};

/// Collects all values in the given deopt state which can't be replaces with
/// lazy objects.
void collectNonReplaceableValues(OperandBundleUse OBU,
                                 SmallPtrSetImpl<Value *> &Result);
/// Checks whether it's legal to replace new allocation values in the deopt 
/// state with symbolic description of lazy materialized object.
/// 
/// The call site must have a deopt state with known deopt format.
bool canReplaceWithLazyObject(OperandBundleUse OBU, const Value *New);
bool canReplaceWithLazyObject(const CallBase *CB, const Value *New);

/// The caller is responsible to check whether the replacement is legal
///
/// Objects is a list of lazy objects.
/// Mapping is a map between values to replace and corresponding lazy objects.
/// Mapping contains the index of the lazy object description in Objects list.
/// There might be more than one value pointing to one lazy object.
bool ReplaceWithLazyObjects(OperandBundleUse OBU, 
                            SmallVectorImpl<LazyObjectBuilder> &Objects,
                            ValueMap<Value*, unsigned> &Mapping,
                            std::vector<Value*> &NewBundle);
/// A convenience wrapper for ReplaceWithLazyObjects which takes one lazy object
/// and one value pointing to this object.
bool ReplaceWithLazyObject(OperandBundleUse OBU, Value *V,
                           LazyObjectBuilder &Object,
                           std::vector<Value*> &NewBundle);

}

#endif
