//===-- FunctionCache.h -----------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This file defines the FunctionCache class.  This class knows how to
// deserialize and serialize 'llvm::Function's (inline candidates,
// specifically) to and from strings.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_FUNCTION_CACHE_H
#define LLVM_AZUL_FUNCTION_CACHE_H

#include <string>
#include <vector>

#include "llvm/ADT/StringRef.h"
#include "llvm/IR/Function.h"
#include "llvm/IR/ValueHandle.h"
#include "llvm/Support/Allocator.h"

namespace llvm {
class Module;
} // namespace llvm

namespace azul {

class SerializedFunction;

namespace internal {
/// The AzulState class holds an instance of the FunctionCache.
///
class FunctionCache {
  // This allocator calls destructors for allocated instance when it
  // dies, so the WeakVH in SerializedFunction doesn't leak.
  llvm::SpecificBumpPtrAllocator<SerializedFunction> allocator;

public:
  /// Factory methods for SerializedFunction instances
  ///
  SerializedFunction *create(llvm::Function *);
  SerializedFunction *create(llvm::StringRef symbol, llvm::StringRef source);
};
} // namespace internal

class SerializedFunction {
  llvm::AssertingVH<llvm::Function> origFunctionHandle;
  llvm::WeakVH functionHandle;
  std::string serializedBody;
  std::string symbol;

  SerializedFunction(llvm::Function *F)
      : origFunctionHandle(F), functionHandle(F), symbol(F->getName().str()) {}

  SerializedFunction(llvm::StringRef symbol, llvm::StringRef source)
      : serializedBody(source.str()), symbol(symbol.str()) {}

public:
  /// This will return an instance of the SerializedFunction for the
  /// module passed in, derserializing it into the Module if needed.
  ///
  llvm::Function *getFunction(llvm::Module *M);

  /// Get the name of the function this represents.
  ///
  const std::string &getSymbol() const { return symbol; }

  /// The serialized body of the function.  This can be parsed by the
  /// AsmParser to get an llvm::Module containing the llvm::Function
  /// this SerializedFunction instance represents.
  ///
  const std::string &getBody();

  friend class internal::FunctionCache;
};
} // namespace azul

#endif // LLVM_AZUL_FUNCTION_CACHE_H
