//===- llvm/Azul/TypeUtils.h - Type Utilities -------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef LLVM_TYPES_UTILS_H
#define LLVM_TYPES_UTILS_H

#include "llvm/ADT/DenseSet.h"

#include "llvm/IR/Orca/Attributes.h"
#include "llvm/IR/InstrTypes.h"
#include "llvm/IR/Instruction.h"
#include "llvm/IR/Module.h"
#include "llvm/IR/Value.h"
#include "llvm/IR/Type.h"

#include <optional>

namespace llvm {
class DominatorTree;
class BasicBlockEdge;
}

namespace azul {
namespace TypeUtils {
constexpr const char *ArrayAccessTBAASuffix = ".array";
constexpr const char *ArrayLengthAccessTBAAName = "array_length_access";
constexpr const char *LocalFieldsTBAAName = "tbaa_local_fields";

/// Represents a compile-time type (e.g. class or interface) ID.
/// These IDs are used to represent types during compilation.
/// These IDs might be different from the IDs used to represent the
/// same types in run-time, e.g. the type IDs embedded in the IR for
/// type checks in run-time.
class CompileTimeKlassID {
private:
  uint64_t ID;

public:
  CompileTimeKlassID() : ID(0) {}
  explicit CompileTimeKlassID(uint64_t ID) : ID(ID) {}

  uint64_t getID() const { return ID; }

  bool operator==(const CompileTimeKlassID &other) const {
    return (ID == other.ID);
  }
};

// Given a CompileTimeKlassID returns a value that represents
// the same type in run time. Note that the run time value might
// be different from the compile time value. All type IDs embedded
// in compiled code must be produced by this function.
std::optional<llvm::Value *>
compileTimeToRunTimeKlassID(llvm::Type *Ty, CompileTimeKlassID KID);

// Given a Value that represents a run time KlassID returns the corresponding 
// CompileTimeKlassID that represents the same type. This function must be used
// when extracting type IDs from the IR.
std::optional<CompileTimeKlassID>
runTimeToCompileTimeKlassID(const llvm::Value *);

inline llvm::raw_ostream &operator<<(llvm::raw_ostream &OS,
                                     const CompileTimeKlassID &T) {
  OS << T.getID();
  return OS;
}

class JavaType {
public:
  // Represents an upper bound on the possible types a value might hold.
  // Note that KlassID can represent either a class or an interface.
  CompileTimeKlassID KlassID;
  // If IsExact is set, value can only hold references to objects of exactly
  // the statically determined type and not a subtype thereof.
  bool IsExact;
  JavaType() : IsExact(false) {}
  JavaType(CompileTimeKlassID KlassID, bool IsExact)
      : KlassID(KlassID), IsExact(IsExact) {}

  void dump() const;
  void print(llvm::raw_ostream &OS) const;

  bool operator==(const JavaType &other) const {
    return (KlassID == other.KlassID) && (IsExact == other.IsExact);
  }

  bool operator!=(const JavaType &other) const { return !(*this == other); }
};

inline llvm::raw_ostream &operator<<(llvm::raw_ostream &OS,
                                     const std::optional<JavaType> &T) {
  if (T)
    T->print(OS);
  else
    OS << "unknown";
  return OS;
}

} // End TypeUtils namespace
} // End azul namespace

template <> struct llvm::DenseMapInfo<azul::TypeUtils::CompileTimeKlassID> {
  static inline azul::TypeUtils::CompileTimeKlassID getEmptyKey() {
    return azul::TypeUtils::CompileTimeKlassID();
  }

  static inline azul::TypeUtils::CompileTimeKlassID getTombstoneKey() {
    return azul::TypeUtils::CompileTimeKlassID((uint64_t)-1);
  }

  static bool isEqual(azul::TypeUtils::CompileTimeKlassID L,
                      azul::TypeUtils::CompileTimeKlassID R) {
    return L == R;
  }

  static unsigned getHashValue(azul::TypeUtils::CompileTimeKlassID Val) {
    return llvm::DenseMapInfo<uint64_t>::getHashValue(Val.getID());
  }
};

template <> struct llvm::DenseMapInfo<azul::TypeUtils::JavaType> {
  static inline azul::TypeUtils::JavaType getEmptyKey() { return {}; }

  static inline azul::TypeUtils::JavaType getTombstoneKey() {
    return {llvm::DenseMapInfo<
                azul::TypeUtils::CompileTimeKlassID>::getTombstoneKey(),
            false};
  }

  static bool isEqual(azul::TypeUtils::JavaType L,
                      azul::TypeUtils::JavaType R) {
    return L == R;
  }

  static unsigned getHashValue(azul::TypeUtils::JavaType Val) {
    return hash_combine(
        llvm::DenseMapInfo<azul::TypeUtils::CompileTimeKlassID>::getHashValue(
            Val.KlassID),
        Val.IsExact);
  }
};

namespace azul {
namespace TypeUtils {

struct EdgeImpliedType {
  llvm::Value *V = nullptr;
  JavaType T;
  EdgeImpliedType() = default;
  EdgeImpliedType(llvm::Value *V, JavaType T) : V(V), T(T) {}
};
/// Returns the type constrained implied if the given edge is taken.
///
/// For example:
///   if (is_subtype_of(42, get_klass_id(X)))
///     ; EdgeImpliedType: X is JavaType(42, non-exact)
///     ...
std::optional<EdgeImpliedType> getEdgeImpliedType(llvm::BasicBlockEdge Edge);

std::optional<JavaType> MeetJavaTypes(llvm::LLVMContext &C,
                                      const std::optional<JavaType> &T1,
                                      const std::optional<JavaType> &T2);

/// The callback used by MeetIncomingTypes to handle an individual operand.
/// If the operand should be skipped returns false. Otherwise returns true
/// and sets the out Optional<JavaType> parameter.
using HandleOpFunctionTy = 
  std::function<bool(llvm::User::const_op_iterator, 
                     std::optional<JavaType> &)>;

/// Meets the types of instruction's operands. Typically of a PHI or a Select.
/// The caller specifies how the operand types should be computed by providing 
/// HandleOp callback.
std::optional<JavaType> MeetIncomingTypes(
    llvm::User::const_op_iterator begin,
    llvm::User::const_op_iterator end,
    HandleOpFunctionTy HandleOp);

/// Combines type facts about the same value coming from different sources.
/// If the given types are incompatible we are analyzing dead code and can
/// return any value.
std::optional<JavaType> JoinJavaTypes(llvm::LLVMContext &C,
                                      const std::optional<JavaType> &T1,
                                      const std::optional<JavaType> &T2);

std::optional<JavaType> getJavaTypeFromAttributes(llvm::AttributeList Attrs,
                                                  int Index);

void setJavaTypeForAttributes(llvm::LLVMContext &C, llvm::AttributeList &AS,
                              unsigned Index, const JavaType &Type);

std::optional<JavaType> getJavaTypeFromMetadata(const llvm::Instruction *I);

/// Creates a JavaType union metadata node.
///   !0 = !{!0, ..., !n}
///   !1 = !{i64 <KlassID>, i1 <IsExact>}
///   ...
///   !n = !{i64 <KlassID>, i1 <IsExact>}
llvm::MDNode *createJavaTypeUnionMetadata(llvm::LLVMContext &C,
                                          llvm::ArrayRef<JavaType> Types);
/// Parses the types from the given JavaType union metadata node.
void parseJavaTypeUnionMetadata(llvm::MDNode *MD,
                                llvm::SmallVectorImpl<JavaType> &Types);

// If available returns JavaType which denotes static type of the value.
// If the CtxI is specified performs context-sensitive analysis and returns
// type of the value at the specified instruction. The DT must be passed in
// this case.
std::optional<JavaType> getJavaType(const llvm::Value *V,
                                    const llvm::Instruction *CtxI = nullptr,
                                    const llvm::DominatorTree *DT = nullptr);

// Returns klass ID if it's available and known to be exact.
std::optional<CompileTimeKlassID> getExactKlassID(const llvm::Value *V);

/// Returns the JavaType of the value passed as an argument to the call.
/// Returns context-sensitive type which holds true at the call instruction.
/// This query uses type information attached via attibutes to the call to
/// improve type results.
std::optional<JavaType>
getArgumentJavaType(llvm::CallBase *CB, unsigned ArgNo,
                    const llvm::DominatorTree *DT = nullptr);

/// Return true if we can prove objects of the given type are definitely
/// arrays.  (NOTE: This currently doesn't handle primitive arrays, and will
/// always return false, which is likely unexpected.)
bool isArrayType(llvm::LLVMContext &C, const JavaType &T);

/// Given a type representing a (possible) array type, return the type of
/// it's elements.  Returns std::nullopt if the type is not an array of objects.
/// If 'InheritExactness' is true returned type will have same IsExact flag as
/// an input array. Otherwise returned type is always non exact.
std::optional<JavaType> getArrayElementType(llvm::LLVMContext &C,
                                            const JavaType &T,
                                            bool InheritExactness = false);

/// A cover function around the JavaType function of the same name which
/// includes finding the type of the 'ArrayV' if possible.
std::optional<JavaType> getArrayElementType(
    const llvm::Value *ArrayV, const llvm::Instruction *CtxI = nullptr,
    const llvm::DominatorTree *DT = nullptr, bool InheritExactness = false);

// Sets metadata representing java type for V if it is an instruction and
// has less specific metadata installed.
bool setMoreSpecificJavaType(llvm::Value *V, const JavaType &Type);

// Sets metadata representing java type for V if it is an instruction.
void setJavaType(llvm::Value *V, const JavaType &Type);

// Return true if this instruction accesses one of the array elements.
// I.e if this is load it loads array element, not array length or header.
bool isArrayElementAccess(const llvm::Instruction *I);

// Return true if this instruction accesses array length.
bool isArrayLengthAccess(const llvm::Instruction *I);

// Return true if instruction accesses one of the java instance fields.
bool isObjectFieldAccess(const llvm::Instruction *I);

// Return true if the given types intersect, i.e. references of the given
// types can point to the same object.
// This is a conservative check, it returns true if can't prove otherwise.
bool canTypesIntersect(llvm::LLVMContext &C, const JavaType &A,
                       const JavaType &B);

std::optional<JavaType> getArgumentJavaTypeAttribute(const llvm::CallBase *CB,
                                                     unsigned ArgNo);

void setArgumentJavaTypeAttribute(llvm::CallBase *CB, unsigned ArgNo,
                                  const JavaType &Type);

std::optional<JavaType> getArgumentJavaTypeAttribute(const llvm::Function *F,
                                                     unsigned ArgNo);

void setArgumentJavaTypeAttribute(llvm::Function *F, unsigned ArgNo,
                                  const JavaType &Type);

std::optional<JavaType>
getReturnValueJavaTypeAttribute(const llvm::CallBase *CB);

void setReturnValueJavaTypeAttribute(llvm::CallBase *CB, const JavaType &Type);

/// Will return false when hoisting a load would require loading a reference
/// typed field at a location where the memory location can not be proven to
/// contain a reference. Will otherwise return true.  Note that this acts as a
/// negative filter only.  Normal dereferenceability rules still apply.
bool isSafeToSpeculateJavaHeapLoad(const llvm::Value *V,
                                   const llvm::Type *LoadTy,
                                   const llvm::DataLayout &DL,
                                   const llvm::Instruction *CtxI = nullptr,
                                   const llvm::DominatorTree *DT = nullptr);

/// Add metadata to \p I based on what we can prove about the instruction.
/// For instance, if \p I is a load instruction that is guaranteed to load a
/// positive value, then this function will annotate \p I with \c !range
/// metadata.
///
/// NB: the location used for proving contraints on \p I is \p I itself.  So,
/// e.g., if LICM wants to call this on an instruction it just hoisted, then
/// it should do so //after// hoisting it.
bool addValidMetadata(llvm::Instruction *I,
                      const llvm::DominatorTree *DT = nullptr);

/// Checks if the value is guaranteed to be a concrete, i.e. non-abstract,
/// class ID
bool isKnownConcreteKlassID(const llvm::Value *V);

/// Return a Value representing the specified arrays length (if it is in fact
/// an array, and we know how to trivially find such a value).
std::optional<llvm::Value *> getArrayLength(const llvm::Value *V);
std::optional<uint64_t> getArrayLengthConstant(const llvm::Value *V);

/// Return true if \p BV is "physically" dereferenceable_or_null for sizeof(\p
/// Ty) bytes starting at offset \p Offset by using inferred Java level type
/// information.  We do not have to worry about emitting a gc pointer typed load
/// resulting in a non-gc pointer at this layer, that is handled by the caller.
bool isDereferenceableOrNullFromJavaType(const llvm::Value *BV,
                                         llvm::APInt SizeInBytes,
                                         const llvm::Instruction *CtxI,
                                         const llvm::DominatorTree *DT);

/// getJavaTypeForKlassID(V) is an upper bound on the possible dynamic types
/// represented by the given KlassID.  Note that the argument is a KlassID, not
/// an instance of a java type. It can work even on incomplete IR, so you can
/// use it from InstructionSimplify.
std::optional<JavaType>
getJavaTypeForKlassID(llvm::Value *KidV,
                      const llvm::Instruction *CtxI = nullptr,
                      const llvm::DominatorTree *DT = nullptr);

/// If an isSubtypeOf call for the given upper bound of the possible dynamic 
/// types for the parent and the child always evaluates to a constant, returns
/// the constant result of the call.
std::optional<bool> isKnownSubtypeOf(llvm::LLVMContext &C,
                                     const JavaType &Parent,
                                     const JavaType &Child);

std::optional<bool> isJavaLangObject(llvm::LLVMContext &C,
                                     CompileTimeKlassID KlassID);
std::optional<bool> isInterface(llvm::LLVMContext &C,
                                CompileTimeKlassID KlassID);
std::optional<bool> isAbstract(llvm::LLVMContext &C,
                               CompileTimeKlassID KlassID);

/// This is a convenience wrapper over the VM callback. Its purpose is to
/// handle some simple cases like Parent == Child without asking the VM. This
/// helps to reduce VM-compiler callback traffic.
std::optional<bool> isSubtypeOf(llvm::LLVMContext &C, CompileTimeKlassID Parent,
                                CompileTimeKlassID Child);

/// Return true if NewT is a more specific type than OrigT
bool isMoreSpecificJavaType(llvm::LLVMContext &C,
                            const std::optional<JavaType> &OrigT,
                            const std::optional<JavaType> &NewT);

bool addBasePointerJavaTypeMDNode(llvm::Instruction *I, llvm::Value *Addr,
                                  llvm::DominatorTree *DT);
} // End TypeUtils namespace
} // End azul namespace

#endif
