//===- llvm/Analysis/Orca/RemoveAllocations.h - -----------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef REMOVEALLOCATIONS_H
#define REMOVEALLOCATIONS_H

#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/MapVector.h"
#include "llvm/ADT/PointerUnion.h"
#include "llvm/ADT/SmallSet.h"
#include "llvm/ADT/TinyPtrVector.h"
#include <optional>

namespace llvm {
class DominatorTree;
class Function;
class Instruction;
class Use;
class User;
class TargetLibraryInfo;
}

namespace azul {
struct RemoveAllocations {
  /// Describes a dependency which prevent elimination of an allocation.
  ///
  /// A removal dependency is either a use or an instruction. To make the
  /// allocation removable a use dependency can be resolved by either removing
  /// the user or rewriting the use. For example, we might be able to
  /// dematerialize deopt state uses. An instruction dependency can only be
  /// resolved by removing the instruction.
  using RemovalDependency =
    llvm::PointerUnion<llvm::Use *, llvm::Instruction *>;

  /// Describes what we know about a potentially removable allocation.
  struct AllocationInfo {
    /// The allocation instruction. It's either an allocation call or a PHI
    /// which has allocations as some of the incoming values.
    llvm::Instruction *Inst;
    /// The set of instructions which will be safely removed once the allocation
    /// is removed.
    llvm::SmallPtrSet<llvm::Instruction *, 16> RemovableUsers;
    /// The set of dependencies which prevent removal of the allocation.
    llvm::SmallSet<RemovalDependency, 4> UnremovableDependencies;

    AllocationInfo(llvm::Instruction *AI);
    void dump() const;
  };
  // MapVector is used to make the iteration order deterministic.
  llvm::MapVector<llvm::Instruction *, AllocationInfo> Allocations;

  RemoveAllocations(llvm::Function &F, llvm::DominatorTree &DT,
                    const llvm::TargetLibraryInfo *TLI)
      : F(F), DT(DT), TLI(TLI) {}
  void collectAllocationDependencies();

  /// Returns the allocation which need to be removed in order to remove the
  /// given dependency.
  std::optional<llvm::Instruction *>
  getDependentAllocation(RemovalDependency Dep);

  /// Perform the actons to remove allocation AI and all its users
  static void removeAllocation(llvm::Instruction *AI, llvm::DominatorTree &DT,
                               const llvm::TargetLibraryInfo *TLI);

  static llvm::User *getDependencyUser(RemovalDependency Dep);

private:
  llvm::Function &F;
  llvm::DominatorTree &DT;
  const llvm::TargetLibraryInfo *TLI;

  llvm::DenseMap<llvm::Instruction *, llvm::Instruction *>
      InstructionToDependentAllocationMap;

  void
  analyzeAllocation(llvm::Instruction *AI,
                    llvm::SmallVectorImpl<llvm::Instruction *> &OuterWorklist);
  void registerDependentAllocation(llvm::Instruction *UserI, 
                                   llvm::Instruction *Allocation);
};
}

#endif /* REMOVEALLOCATIONS_H */
