//===- llvm/Analysis/Orca/JavaTypeAnalysis.h - JavaTypeAnalysis -*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2020 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef JAVATYPEANALYSIS_H
#define JAVATYPEANALYSIS_H

#include "llvm/IR/AssemblyAnnotationWriter.h"
#include "llvm/IR/PassManager.h"
#include <optional>

namespace llvm {
class BasicBlock;
class BasicBlockEdge;
class DominatorTree;
class Function;
class Instruction;
class Value;
class raw_ostream;

struct JavaTypeAnalysisPrinterPass : PassInfoMixin<JavaTypeAnalysisPrinterPass> {
  raw_ostream &OS = dbgs();
  JavaTypeAnalysisPrinterPass() = default;
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);
};
}

namespace azul {
namespace TypeUtils {
class JavaType;
}

class JavaTypeAnalysis {
  friend class JavaTypeAnalysisAnnotatedWriter;

  struct BasicBlockState {
    llvm::DenseMap<llvm::Value *, std::optional<TypeUtils::JavaType>>
        KnownTypes;
    std::optional<TypeUtils::JavaType> getCachedJavaType(llvm::Value *V);
    std::optional<TypeUtils::JavaType> getJavaType(llvm::Value *V);
    bool empty() const;
    void print(llvm::raw_ostream &OS, llvm::StringRef Prefix) const;
  };

  llvm::DenseMap<llvm::BasicBlock *, BasicBlockState> States;

  llvm::Function &F;
  llvm::DominatorTree &DT;

public:
  JavaTypeAnalysis(llvm::Function &F, llvm::DominatorTree &DT) : F(F), DT(DT) {}

  std::optional<TypeUtils::JavaType>
  getJavaType(llvm::Value *V, llvm::Instruction *CtxI = nullptr);

  void analyze();

private:
  BasicBlockState getState(const llvm::BasicBlock *BB);
  std::optional<TypeUtils::JavaType>
  getCachedJavaType(llvm::Value *V, llvm::Instruction *CtxI);
  BasicBlockState getStateForEdge(llvm::BasicBlockEdge Edge);
  void visitBlock(llvm::BasicBlock *BB);
};

class JavaTypeAnalysisAnnotatedWriter : public llvm::AssemblyAnnotationWriter {
  JavaTypeAnalysis &JTA;

public:
  JavaTypeAnalysisAnnotatedWriter(JavaTypeAnalysis &JTA) : JTA(JTA) {}

  virtual void emitBasicBlockStartAnnot(const llvm::BasicBlock *,
                                        llvm::formatted_raw_ostream &) override;

};

}

#endif /* JAVATYPEANALYSIS_H */
