//===- llvm/Orca/CodeLatencyEstimator.h -------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2023 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This file provides the interface for Azul's own CodeLatencyEstimator pass.
// This pass intends to provide upper bound latency estimation
// for a given piece of IR.
///
//===----------------------------------------------------------------------===//

#ifndef CODE_LATENCY_ESTIMATOR_H
#define CODE_LATENCY_ESTIMATOR_H

#include "llvm/IR/PassManager.h"

namespace llvm {

class Loop;
class LoopInfo;
class ScalarEvolution;
class LoopBodyNode;

class CodeLatencyEstimator {
  static constexpr unsigned MaxEstimateBitWidth = 64;

  // Note: all latency calculations in this class generally should be in
  // uint64_t, but this specific alias should be used only for latency values to
  // avoid confusion.
public:
  using LatencyValue = uint64_t;

  // When the value is std::nullopt, it means that the estimate exceeds what
  // LatencyValue's underlying type could fit.
  // Please don't use this alias for anything other than latency estimates
  // (i.e. when you just need an std::optional<uint64_t>) to avoid confusion.
  using LatencyEstimate = std::optional<LatencyValue>;

private:
  Function &F;
  ScalarEvolution &SE;
  LoopInfo &LI;

  DenseMap<Loop *, LatencyEstimate> LoopIterationLatencyCache;

  /// Returns Some if upper limit for backedge-taken count of the loop is known
  /// and it fits into uint64_t, std::nullopt otherwise.
  std::optional<uint64_t> getBackedgeTakenCount(Loop *L);

  /// Returns the number of times loop header is taken if we enter the loop.
  std::optional<uint64_t> getLoopIterationCount(Loop *L);

  LatencyEstimate getLegacyLoopIterationLatencyEstimate(Loop *L);

  /// NOTE: This function does only the calculations, it doesn't cache the
  ///       result. It's intended to be used only by the wrapper that handles
  ///       the caching.
  LatencyEstimate estimateLoopIterationLatency(Loop *L);

public:
  explicit CodeLatencyEstimator(Function &F, ScalarEvolution &SE, LoopInfo &LI)
      : F(F), SE(SE), LI(LI) {}

  LatencyEstimate getNodeLatency(const LoopBodyNode &Node);

  /// Call it when a loop is changed.
  /// NOTE: CLE relies on ScalarEvolution and LoopInfo and it expects its users
  ///       to keep those analyses up to date.
  void forgetLoop(Loop *L);

  LatencyEstimate getBlockLatency(BasicBlock *BB) {
    assert(BB);
    return 1;
  }

  /// A caching wrapper for `estimateLoopIterationLatency`, just to split
  /// calculations and caching.
  LatencyEstimate getLoopIterationLatency(Loop *L);

  /// The estimate is the latency of an iteration multiplied by maximum number
  /// of iterations.
  LatencyEstimate getLoopLatency(Loop *L);

  /// Returns the maximum number of iterations in which the loop won't reach
  /// the given latency threshold. 0 means we can't guarantee that even one
  /// iteration won't reach the threshold.
  uint64_t getIterationsUntilLatencyThreshold(Loop *L,
                                              LatencyValue LatencyThreshold);

  /// Returns false if latency of the loop is known to be below the given
  /// threshold, true otherwise.
  bool canLoopExceedLatencyThreshold(Loop *L, LatencyValue LatencyThreshold);

  void print(raw_ostream &OS);
  void annotateLoop(Loop *L, raw_ostream &OS, StringRef Prefix = "");
  void dump();
};

class CodeLatencyEstimatorPrinterPass
    : public PassInfoMixin<CodeLatencyEstimatorPrinterPass> {
public:
  CodeLatencyEstimatorPrinterPass() = default;

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &AM);
};
} // namespace llvm

#endif /* CODE_LATENCY_ESTIMATOR_H */
