//===- llvm/Analysis/Orca/AvailableDefTracker.h - DefTracker ----*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef AVAILABLEDEFTRACKER_H
#define AVAILABLEDEFTRACKER_H

#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/SmallPtrSet.h"
#include "llvm/IR/AssemblyAnnotationWriter.h"
#include "llvm/IR/PassManager.h"

namespace llvm {
class AAResults;
struct AvailableDefs;
class BatchAAResults;
class CallBase;
class BasicBlock;
class Function;
class Value;
class InlineFunctionInfo;
class Instruction;
class MemorySSA;
class formatted_raw_ostream;
class raw_ostream;

struct AvailableDefPrinterPass : PassInfoMixin<AvailableDefPrinterPass> {
  raw_ostream &OS = dbgs();
  AvailableDefPrinterPass() = default;
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);
};
}

namespace azul {
class AvailableDefTracker {
  friend class AvailableDefAnnotatedWriter;
private:
  /// The caller state AvailableDefTracker computes and maintains.
  /// Essentially contains the set of store instructions which have not yet
  /// been clobbered thus can be used in callees as available defs.
  struct CallerDefs {
    /// Set of stores which haven't been clobbered yet.
    llvm::SmallPtrSet<llvm::Instruction *, 8> Defs;
    /// An invariant subset of Defs set.
    llvm::SmallPtrSet<llvm::Instruction *, 8> InvariantDefs;

    /// A map from an address to the defs corresponding to this address
    /// There is only one def per the address unless dead code is involved.
    llvm::DenseMap<llvm::Value *, llvm::SmallPtrSet<llvm::Instruction *, 2>>
        TrackedPointers;
    
    void add(llvm::Instruction *I);
    void markInvariant(llvm::Instruction *I);
    void erase(llvm::Instruction *I);
    void clearNonInvariant();
    void meet(CallerDefs &Other);

    bool empty() { return Defs.empty(); }

    void verify();
    void print(llvm::raw_ostream &OS);
  };

  llvm::Function *Caller;
  llvm::DenseMap<llvm::CallBase *, CallerDefs> CallDefs;

  llvm::DenseMap<llvm::BasicBlock *, CallerDefs> BBEndDefs;

  void invalidate();

  CallerDefs computeCallerBBEntryState(llvm::BasicBlock *BB);
  void scanCallerBB(llvm::BasicBlock *BB, CallerDefs &Defs,
                    llvm::BatchAAResults &CallerAA);
  void analyzeCallerBB(llvm::BasicBlock *BB, llvm::BatchAAResults &CallerAA);

public:
  AvailableDefTracker(llvm::Function *Caller) : Caller(Caller){};
  void collectCallerDefs(llvm::AAResults &CallerAA);
  void collectAvailableDefs(const llvm::CallBase *Call,
                            llvm::MemorySSA &CalleeMSSA,
                            llvm::AvailableDefs &Defs) const;

  /// Notifies tracker about inlining into the caller.
  void callSiteInlined(llvm::CallBase *Call, llvm::InlineFunctionInfo &IFI);
};

class AvailableDefAnnotatedWriter : public llvm::AssemblyAnnotationWriter {
  AvailableDefTracker &T;
  llvm::FunctionAnalysisManager &FAM;

public:
  AvailableDefAnnotatedWriter(AvailableDefTracker &T,
                              llvm::FunctionAnalysisManager &FAM)
      : T(T), FAM(FAM) {}

  virtual void emitInstructionAnnot(const llvm::Instruction *I,
                                    llvm::formatted_raw_ostream &OS) override;
  virtual void emitBasicBlockEndAnnot(const llvm::BasicBlock *,
                                      llvm::formatted_raw_ostream &) override;
  
};
}

#endif /* AVAILABLEDEFTRACKER_H */
