import sys
import os

from lldb import LLDB_INVALID_ADDRESS
from lldb import SBValue, SBData, SBDebugger, SBError
from lldb import eBasicTypeLong, eBasicTypeUnsignedLong, eBasicTypeUnsignedChar, eBasicTypeChar

#################################################################################################################
# This file contains two kinds of pretty-printers: summary and synthetic.
#
# Important classes from LLDB module:
#   SBValue: the value of a variable, a register, or an expression
#   SBType:  the data type; each SBValue has a corresponding SBType
#
# Summary provider is a function with the type `(SBValue, dict) -> str`.
#   The first parameter is the object encapsulating the actual variable being displayed;
#   The second parameter is an internal support parameter used by LLDB, and you should not touch it.
#
# Synthetic children is the way to provide a children-based user-friendly representation of the object's value.
# Synthetic provider is a class that implements the following interface:
#
#     class SyntheticChildrenProvider:
#         def __init__(self, SBValue, dict)
#         def num_children(self)
#         def get_child_index(self, str)
#         def get_child_at_index(self, int)
#         def update(self)
#         def has_children(self)
#         def get_value(self)
#
#
# You can find more information and examples here:
#   1. https://lldb.llvm.org/varformats.html
#   2. https://lldb.llvm.org/python-reference.html
#   3. https://lldb.llvm.org/python_reference/lldb.formatters.cpp.libcxx-pysrc.html
#   4. https://github.com/llvm-mirror/lldb/tree/master/examples/summaries/cocoa
################################################################################################################

PY3 = sys.version_info[0] == 3
if PY3:
    from typing import Optional, List


def unwrap_non_null(nonnull):
    # type: (SBValue) -> SBValue
    """
    struct NonNull<T> { pointer: *const T }
    """
    return nonnull.GetChildMemberWithName("pointer")


def get_inner_rc_data_ptr(valobj):
    # type: (SBValue) -> SBValue
    return valobj.GetChildMemberWithName("data_ptr").GetChildAtIndex(2).AddressOf()


def unwrap_unique_or_non_null(unique_or_nonnull):
    # type: (SBValue) -> SBValue
    """
    rust 1.33.0: struct Unique<T: ?Sized> { pointer: *const T, ... }
    rust 1.62.0: struct Unique<T: ?Sized> { pointer: NonNull<T>, ... }
    struct NonNull<T> { pointer: *const T }
    """
    ptr = unique_or_nonnull.GetChildMemberWithName("pointer")
    inner_ptr = ptr.GetChildMemberWithName("pointer")
    if inner_ptr.IsValid():
        return inner_ptr
    else:
        return ptr


def get_template_params(type_name):
    # type: (str) -> List[str]
    params = []
    level = 0
    start = 0
    for i, c in enumerate(type_name):
        if c == '<':
            level += 1
            if level == 1:
                start = i + 1
        elif c == '>':
            level -= 1
            if level == 0:
                params.append(type_name[start:i].strip())
        elif c == ',' and level == 1:
            params.append(type_name[start:i].strip())
            start = i + 1
    return params


def get_max_string_summary_length(debugger):
    # type: (SBDebugger) -> int
    debugger_name = debugger.GetInstanceName()
    max_len = SBDebugger.GetInternalVariableValue("target.max-string-summary-length", debugger_name)
    return int(max_len.GetStringAtIndex(0))


def _repr(c):
    # https://doc.rust-lang.org/reference/tokens.html#ascii-escapes
    if c == '\n':
        return '\\n'
    if c == '\r':
        return '\\r'
    if c == '\t':
        return '\\t'
    if c == '\\':
        return '\\\\'
    if c == '\0':
        return '\\0'
    if c == '"':
        return '\\"'

    char_code = ord(c)
    if char_code >= 0 and (char_code <= 0x1f or char_code == 0x7f):
        return "\\x%02X" % char_code

    return c


def escape_bytes(s):
    return ''.join([_repr(c) for c in s])


def read_raw_string(data_ptr, length):
    # type: (SBValue, int) -> str
    if data_ptr is None or length == 0:
        return '""'

    max_string_summary_length = get_max_string_summary_length(data_ptr.GetTarget().GetDebugger())
    length_to_read = min(length, max_string_summary_length)

    process = data_ptr.GetProcess()
    start = data_ptr.GetValueAsUnsigned()
    error = SBError()
    data = process.ReadMemory(start, length_to_read, error)
    data = data.decode(encoding='UTF-8', errors='replace') if PY3 else data
    data = escape_bytes(data)

    return '"%s"' % data


def get_vec_data_ptr(valobj):
    # type: (SBValue) -> SBValue
    buf = valobj.GetChildMemberWithName("buf")
    inner = buf.GetChildMemberWithName("inner")
    if inner.IsValid():
        return unwrap_unique_or_non_null(inner.GetChildMemberWithName("ptr"))
    else:
        return unwrap_unique_or_non_null(buf.GetChildMemberWithName("ptr"))


def get_vec_length(valobj):
    # type: (SBValue) -> int
    return valobj.GetChildMemberWithName("len").GetValueAsUnsigned()


def extract_data_and_len_from_ffi_string(valobj):
    # type: (SBValue) -> tuple[Optional[SBValue], int]
    process = valobj.GetProcess()
    error = SBError()
    slice_ptr = valobj.GetLoadAddress()
    if slice_ptr == LLDB_INVALID_ADDRESS:
        return None, 0
    char_ptr_type = valobj.GetTarget().GetBasicType(eBasicTypeChar).GetPointerType()
    data_ptr = valobj.CreateValueFromAddress('start', slice_ptr, char_ptr_type)
    length = process.ReadPointerFromMemory(slice_ptr + process.GetAddressByteSize(), error)
    return data_ptr, length

def get_template_argument_type_nt_workaround(struct_type_name, target):
    start = struct_type_name.find("<") + 1
    end = struct_type_name.rfind(",")
    return target.FindFirstType(struct_type_name[start:end])

def get_rawvec_element_type(valobj):
    buf = valobj.GetChildMemberWithName("buf")
    if not buf.IsValid():
        return None
    marker = buf.GetChildMemberWithName("_marker")
    if not marker.IsValid():
        return None
    # FIXME: On Windows, `GetTemplateArgumentType` crashes the whole debugger
    if os.name != 'nt':
        if valobj.TypeIsPointerType():
            base_type = valobj.Dereference().GetType()
        else:
            base_type = valobj.GetType()
        return base_type.GetTemplateArgumentType(0).GetCanonicalType()
    else:
        return get_template_argument_type_nt_workaround(valobj.GetType().name, valobj.target)

class ValueBuilder:
    def __init__(self, valobj):
        # type: (SBValue) -> None
        self.valobj = valobj
        process = valobj.GetProcess()
        self.endianness = process.GetByteOrder()
        self.pointer_size = process.GetAddressByteSize()

    def from_int(self, name, value):
        # type: (str, int) -> SBValue
        type = self.valobj.GetTarget().GetBasicType(eBasicTypeLong)
        data = SBData.CreateDataFromSInt64Array(self.endianness, self.pointer_size, [value])
        return self.valobj.CreateValueFromData(name, data, type)

    def from_uint(self, name, value):
        # type: (str, int) -> SBValue
        type = self.valobj.GetTarget().GetBasicType(eBasicTypeUnsignedLong)
        data = SBData.CreateDataFromUInt64Array(self.endianness, self.pointer_size, [value])
        return self.valobj.CreateValueFromData(name, data, type)


def OptionResultSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    if valobj.GetNumChildren() == 1:
        child = valobj.GetChildAtIndex(0)
        child_summary = child.GetSummary()
        if child_summary is not None:
            return child_summary
        child_value = child.GetValue()
        if child_value is not None:
            return child_value
    if valobj.GetType().GetName().endswith("::None"):
        return "None"
    return ""


def SizeSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return 'size=' + str(valobj.GetNumChildren())


def StdStringSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    vec_non_synth = valobj.GetChildAtIndex(0).GetNonSyntheticValue()
    data_ptr = get_vec_data_ptr(vec_non_synth)
    length = get_vec_length(vec_non_synth)
    return read_raw_string(data_ptr, length)


def StdOsStringSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    buf = valobj.GetChildAtIndex(0).GetChildAtIndex(0)
    is_windows = "Wtf8Buf" in buf.type.name
    vec = buf.GetChildAtIndex(0) if is_windows else buf
    vec_non_synth = vec.GetNonSyntheticValue()
    data_ptr = get_vec_data_ptr(vec_non_synth)
    length = get_vec_length(vec_non_synth)
    return read_raw_string(data_ptr, length)


def StdPathBufSummaryProvider(valobj, _dict):
    return StdOsStringSummaryProvider(valobj.GetChildAtIndex(0), _dict)


def RcInnerStdStrSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    data_ptr = get_inner_rc_data_ptr(valobj)
    length = valobj.GetChildMemberWithName("length").GetValueAsUnsigned()
    return read_raw_string(data_ptr, length)


def StdStrSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    data_ptr = valobj.GetChildMemberWithName("data_ptr")
    length = valobj.GetChildMemberWithName("length").GetValueAsUnsigned()
    return read_raw_string(data_ptr, length)


def StdOsStrPathSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    data_ptr, length = extract_data_and_len_from_ffi_string(valobj)
    return read_raw_string(data_ptr, length)


def StdCStringSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    data_ptr, length = extract_data_and_len_from_ffi_string(valobj)
    return read_raw_string(data_ptr, length - 1)


class ArrayLikeSyntheticProviderBase:
    def __init__(self, valobj, _dict):
        # type: (SBValue, dict) -> None
        self.valobj = valobj
        self.update()

    def get_data_ptr(self):
        # type: () -> SBValue
        pass

    def get_length(self):
        # type: () -> int
        pass

    def get_inner_type(self):
        # type: () -> SBType
        pass

    def num_children(self):
        # type: () -> int
        return self.length

    def get_child_index(self, name):
        # type: (str) -> int
        index = name.lstrip('[').rstrip(']')
        if index.isdigit():
            return int(index)
        else:
            return -1

    def get_child_at_index(self, index):
        # type: (int) -> SBValue
        offset = index * self.element_type_size
        return self.data_ptr.CreateChildAtOffset("[%s]" % index, offset, self.element_type)

    def update(self):
        # type: () -> None
        self.data_ptr = self.get_data_ptr()
        self.length = self.get_length()
        t = get_rawvec_element_type(self.valobj)
        if t is None:
            t = self.data_ptr.GetType().GetPointeeType()
        self.element_type = t
        self.element_type_size = self.element_type.GetByteSize()

    def has_children(self):
        # type: () -> bool
        return True


class RcInnerStdSliceSyntheticProvider(ArrayLikeSyntheticProviderBase):
    def get_data_ptr(self):
        # type: () -> SBValue
        return get_inner_rc_data_ptr(self.valobj)

    def get_length(self):
        # type: () -> int
        return self.valobj.GetChildMemberWithName("length").GetValueAsUnsigned()


class StdSliceSyntheticProvider(ArrayLikeSyntheticProviderBase):
    def get_data_ptr(self):
        # type: () -> SBValue
        return self.valobj.GetChildMemberWithName("data_ptr")

    def get_length(self):
        # type: () -> int
        return self.valobj.GetChildMemberWithName("length").GetValueAsUnsigned()


class StdVecSyntheticProvider(ArrayLikeSyntheticProviderBase):
    """Pretty-printer for alloc::vec::Vec<T>

    struct Vec<T> { buf: RawVec<T>, len: usize }
    struct RawVec<T> { ptr: Unique<T>, cap: usize, ... }
    rust 1.33.0: struct Unique<T: ?Sized> { pointer: *const T, ... }
    rust 1.62.0: struct Unique<T: ?Sized> { pointer: NonNull<T>, ... }
    struct NonNull<T> { pointer: *const T }
    """

    def get_data_ptr(self):
        # type: () -> SBValue
        return get_vec_data_ptr(self.valobj)

    def get_length(self):
        # type: () -> int
        return get_vec_length(self.valobj)

    def get_inner_type(self):
        return get_rawvec_element_type(self.valobj)

class StdVecDequeSyntheticProvider:
    """Pretty-printer for alloc::collections::vec_deque::VecDeque<T>

    struct VecDeque<T> { tail: usize, head: usize, buf: RawVec<T> }
    Since 1.67:
    struct VecDeque<T> { head: usize, len: usize, buf: RawVec<T> }
    """

    def __init__(self, valobj, _dict):
        # type: (SBValue, dict) -> None
        self.valobj = valobj
        self.update()

    def num_children(self):
        # type: () -> int
        return self.len

    def get_child_index(self, name):
        # type: (str) -> int
        index = name.lstrip('[').rstrip(']')
        if not index.isdigit():
            return -1
        index = int(index)
        if 0 <= index and index < self.len:
            return index
        else:
            return -1

    def get_child_at_index(self, index):
        # type: (int) -> SBValue
        start = self.data_ptr.GetValueAsUnsigned()
        address = start + (self.base + index) % self.cap * self.element_type_size
        return self.data_ptr.CreateValueFromAddress("[%s]" % index, address, self.element_type)

    def update(self):
        # type: () -> None
        head = self.valobj.GetChildMemberWithName("head").GetValueAsUnsigned()
        buf = self.valobj.GetChildMemberWithName("buf")
        self.cap = self.compute_cap(buf)
        tail = self.valobj.GetChildMemberWithName("tail")
        if tail: # old layout
            self.base = tail.GetValueAsUnsigned()
            dist = head - self.base
            self.len = dist if dist >= 0 else self.cap + dist
        else: # new layout
            self.base = head
            self.len = self.valobj.GetChildMemberWithName("len").GetValueAsUnsigned()
        inner = buf.GetChildMemberWithName("inner")
        if inner:
            data_ptr = inner.GetChildMemberWithName("ptr")
        else:
            data_ptr = buf.GetChildMemberWithName("ptr")
        self.data_ptr = unwrap_unique_or_non_null(data_ptr)
        self.element_type = get_rawvec_element_type(self.valobj)
        if self.element_type is None:
            self.element_type = self.data_ptr.GetType().GetPointeeType()
        self.element_type_size = self.element_type.GetByteSize()

    def has_children(self):
        # type: () -> bool
        return True

    def compute_cap(self, buf):
        cap = buf.GetChildMemberWithName("cap")
        if not(cap.IsValid()):
            cap = buf.GetChildMemberWithName("inner").GetChildMemberWithName("cap")
        cap_niched = cap.GetChildAtIndex(0)
        if cap_niched: # since https://github.com/rust-lang/rust/commit/502df1b7d4f3ba61d55cf2b3a47785a7a4df940b
            cap = cap_niched
        return cap.GetValueAsUnsigned()

class StdHashMapSyntheticProvider:
    """Pretty-printer for hashbrown's HashMap"""

    def __init__(self, valobj, _dict, show_values=True):
        # type: (SBValue, dict, bool) -> None
        self.valobj = valobj
        self.show_values = show_values
        self.update()

    def num_children(self):
        # type: () -> int
        return self.size

    def get_child_index(self, name):
        # type: (str) -> int
        index = name.lstrip('[').rstrip(']')
        if index.isdigit():
            return int(index)
        else:
            return -1

    def get_child_at_index(self, index):
        # type: (int) -> SBValue
        self.update_up_to(index)
        pairs_start = self.data_ptr.GetValueAsUnsigned()
        idx = self.valid_indices[index]
        if self.new_layout:
            idx = -(idx + 1)
        address = pairs_start + idx * self.pair_type_size
        element = self.data_ptr.CreateValueFromAddress("[%s]" % index, address, self.pair_type)
        if self.show_values:
            return element
        else:
            key = element.GetChildAtIndex(0)
            return self.valobj.CreateValueFromData("[%s]" % index, key.GetData(), key.GetType())

    def update(self):
        # type: () -> None
        table = self.table()
        inner_table = table.GetChildMemberWithName("table")

        self.capacity = inner_table.GetChildMemberWithName("bucket_mask").GetValueAsUnsigned() + 1
        self.ctrl = inner_table.GetChildMemberWithName("ctrl").GetChildAtIndex(0)

        self.size = inner_table.GetChildMemberWithName("items").GetValueAsUnsigned()

        if table.type.GetNumberOfTemplateArguments() > 0:
            self.pair_type = table.type.template_args[0].GetTypedefedType()
        else:
            # MSVC LLDB (does not support template arguments at the moment)
            type_name = table.type.name  # expected "RawTable<tuple$<K,V>,alloc::alloc::Global>"
            first_template_arg = get_template_params(type_name)[0]
            self.pair_type = table.GetTarget().FindFirstType(first_template_arg)

        self.pair_type_size = self.pair_type.GetByteSize()

        self.new_layout = not inner_table.GetChildMemberWithName("data").IsValid()
        if self.new_layout:
            self.data_ptr = self.ctrl.Cast(self.pair_type.GetPointerType())
        else:
            self.data_ptr = inner_table.GetChildMemberWithName("data").GetChildAtIndex(0)

        self.u8_type = self.valobj.GetTarget().GetBasicType(eBasicTypeUnsignedChar)
        self.u8_type_size = self.valobj.GetTarget().GetBasicType(eBasicTypeUnsignedChar).GetByteSize()

        self.valid_indices = []
        self.updated_to_idx = 0

    def update_up_to(self, index):
        while self.updated_to_idx < self.capacity and index >= len(self.valid_indices):
            idx = self.updated_to_idx
            address = self.ctrl.GetValueAsUnsigned() + idx * self.u8_type_size
            value = self.ctrl.CreateValueFromAddress("ctrl[%s]" % idx, address,
                                                     self.u8_type).GetValueAsUnsigned()
            is_present = value & 128 == 0
            if is_present:
                self.valid_indices.append(idx)

            self.updated_to_idx += 1

    def table(self):
        # type: () -> SBValue
        if self.show_values:
            hashbrown_hashmap = self.valobj.GetChildMemberWithName("base")
        else:
            # HashSet wraps `hashbrown::HashSet`, which wraps `hashbrown::HashMap`
            hashbrown_hashmap = self.valobj.GetChildAtIndex(0).GetChildAtIndex(0)
        return hashbrown_hashmap.GetChildMemberWithName("table")

    def has_children(self):
        # type: () -> bool
        return True


class StdHashSetSyntheticProvider(StdHashMapSyntheticProvider):
    def __init__(self, valobj, _dict):
        super().__init__(valobj, _dict, show_values=False)


def StdRcSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    strong = valobj.GetChildMemberWithName("strong").GetValueAsUnsigned()
    weak = valobj.GetChildMemberWithName("weak").GetValueAsUnsigned()
    return "strong={}, weak={}".format(strong, weak)


class StdRcSyntheticProvider:
    """Pretty-printer for alloc::rc::Rc<T> and alloc::sync::Arc<T>

    struct Rc<T> { ptr: NonNull<RcInner<T>>, ... }
    struct NonNull<T> { pointer: *const T }
    struct RcInner<T> { strong: Cell<usize>, weak: Cell<usize>, value: T }; note: RcBox in <1.66
    struct Cell<T> { value: UnsafeCell<T> }
    struct UnsafeCell<T> { value: T }

    struct Arc<T> { ptr: NonNull<ArcInner<T>>, ... }
    struct ArcInner<T> { strong: atomic::AtomicUsize, weak: atomic::AtomicUsize, data: T }
    struct AtomicUsize { v: UnsafeCell<usize> }

    Differs for slice and dyn for both Rc and Arc:
    struct ArcInner|RcInner<[T]> { data_ptr: { strong: ..., weak: ..., data: T }, length: usize }
    struct ArcInner|RcInner<dyn T> { pointer: { strong: ..., weak: ..., data: T }, vtable: *const [...] }
    """

    def __init__(self, valobj, _dict, is_atomic=False):
        # type: (SBValue, dict, bool) -> None
        self.valobj = valobj
        self.value_name = "data" if is_atomic else "value"
        original_ptr = unwrap_non_null(self.valobj.GetChildMemberWithName("ptr"))
        maybe_value = original_ptr.GetChildMemberWithName(self.value_name)
        if maybe_value:
            self.ptr = original_ptr
            self.value = maybe_value
        else:
            # dyn ("pointer") or slice/str ("data_ptr"):
            # We'll handle it via a separate synth/summary provider to support &str and &[] properly
            self.ptr = original_ptr.GetChildAtIndex(0)
            self.value = original_ptr.CreateValueFromData(self.value_name, original_ptr.GetData(), original_ptr.GetType())

        self.strong = self.ptr.GetChildMemberWithName("strong").GetChildAtIndex(0).GetChildMemberWithName("value")
        self.weak = self.ptr.GetChildMemberWithName("weak").GetChildAtIndex(0).GetChildMemberWithName("value")
        self.value_builder = ValueBuilder(valobj)
        self.update()

    def num_children(self):
        # type: () -> int
        # Actually there are 3 children, but only the `value` should be shown as a child
        return 1

    def get_child_index(self, name):
        # type: (str) -> int
        if name == self.value_name:
            return 0
        if name == "strong":
            return 1
        if name == "weak":
            return 2
        return -1

    def get_child_at_index(self, index):
        # type: (int) -> Optional[SBValue]
        if index == 0:
            return self.value
        if index == 1:
            return self.strong
        if index == 2:
            return self.value_builder.from_uint("weak", self.weak_count)

        return None

    def update(self):
        # type: () -> None
        self.weak_count = self.weak.GetValueAsUnsigned() - 1

    def has_children(self):
        # type: () -> bool
        return True


class StdArcSyntheticProvider(StdRcSyntheticProvider):
    def __init__(self, valobj, _dict):
        super().__init__(valobj, _dict, is_atomic=True)


class StdCellSyntheticProvider:
    """Pretty-printer for std::cell::Cell"""

    def __init__(self, valobj, _dict):
        # type: (SBValue, dict) -> None
        self.valobj = valobj
        self.value = valobj.GetChildMemberWithName("value").GetChildAtIndex(0)

    def num_children(self):
        # type: () -> int
        return 1

    def get_child_index(self, name):
        # type: (str) -> int
        if name == "value":
            return 0
        return -1

    def get_child_at_index(self, index):
        # type: (int) -> Optional[SBValue]
        if index == 0:
            return self.value
        return None

    def update(self):
        # type: () -> None
        pass

    def has_children(self):
        # type: () -> bool
        return True


def StdRefSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    borrow = valobj.GetChildMemberWithName("borrow").GetValueAsSigned()
    return "borrow={}".format(borrow) if borrow >= 0 else "borrow_mut={}".format(-borrow)


class StdRefSyntheticProvider:
    """Pretty-printer for std::cell::Ref, std::cell::RefMut, and std::cell::RefCell"""

    def __init__(self, valobj, _dict, is_cell=False):
        # type: (SBValue, dict, bool) -> None
        self.valobj = valobj

        borrow = valobj.GetChildMemberWithName("borrow")
        value = valobj.GetChildMemberWithName("value")
        if is_cell:
            self.borrow = borrow.GetChildMemberWithName("value").GetChildMemberWithName("value")
            self.value = value.GetChildMemberWithName("value")
        else:
            self.borrow = borrow.GetChildMemberWithName("borrow").GetChildMemberWithName(
                "value").GetChildMemberWithName("value")
            # BACKCOMPAT: Rust 1.62.0. Drop `else`-branch
            if value.GetChildMemberWithName("pointer"):
                # Since Rust 1.63.0, `Ref` and `RefMut` use `value: NonNull<T>` instead of `value: &T`
                # https://github.com/rust-lang/rust/commit/d369045aed63ac8b9de1ed71679fac9bb4b0340a
                # https://github.com/rust-lang/rust/commit/2b8041f5746bdbd7c9f6ccf077544e1c77e927c0
                self.value = unwrap_unique_or_non_null(value).Dereference()
            else:
                self.value = value.Dereference()

        self.value_builder = ValueBuilder(valobj)

        self.update()

    def num_children(self):
        # type: () -> int
        # Actually there are 2 children, but only the `value` should be shown as a child
        return 1

    def get_child_index(self, name):
        if name == "value":
            return 0
        if name == "borrow":
            return 1
        return -1

    def get_child_at_index(self, index):
        # type: (int) -> Optional[SBValue]
        if index == 0:
            return self.value
        if index == 1:
            return self.value_builder.from_int("borrow", self.borrow_count)
        return None

    def update(self):
        # type: () -> None
        self.borrow_count = self.borrow.GetValueAsSigned()

    def has_children(self):
        # type: () -> bool
        return True


class StdRefCellSyntheticProvider(StdRefSyntheticProvider):
    def __init__(self, valobj, _dict):
        super().__init__(valobj, _dict, is_cell=True)


def StdNonZeroNumberSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    objtype = valobj.GetType()
    inner_field = objtype.GetFieldAtIndex(0)
    inner_element = valobj.GetChildMemberWithName(inner_field.name)

    # before Rust 1.79.0 value was stored as a plain field
    inner_element_value = inner_element.GetValue()
    if inner_element_value is not None:
        return inner_element_value

    # since Rust 1.79.0, value is stored in an inner struct with a plain field

    field = inner_element.GetType().GetFieldAtIndex(0)
    element = inner_element.GetChildMemberWithName(field.name)
    return element.GetValue()


def StdRangeSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return "{}..{}".format(valobj.GetChildMemberWithName("start").GetValueAsSigned(),
                           valobj.GetChildMemberWithName("end").GetValueAsSigned())


def StdRangeFromSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return "{}..".format(valobj.GetChildMemberWithName("start").GetValueAsSigned())


def StdRangeInclusiveSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return "{}..={}".format(valobj.GetChildMemberWithName("start").GetValueAsSigned(),
                            valobj.GetChildMemberWithName("end").GetValueAsSigned())


def StdRangeToSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return "..{}".format(valobj.GetChildMemberWithName("end").GetValueAsSigned())


def StdRangeToInclusiveSummaryProvider(valobj, _dict):
    # type: (SBValue, dict) -> str
    return "..={}".format(valobj.GetChildMemberWithName("end").GetValueAsSigned())
