import {Range, TypeScriptCustomCommandArguments} from "tsc-ide-plugin/protocol"
import {LspProcessingContext, tryHandleCustomTsServerCommandLsp} from "tsc-ide-plugin/ide-commands"
import {type ReverseMapper} from "tsc-ide-plugin/ide-get-element-type"
import {type DocumentsAndMap, getGeneratedRange, getSourceRange} from '@volar/language-service/lib/utils/featureWorkers';
import {asFileName, asUri, getDocs, getLanguageService} from "./volar-utils";
import type {LanguageServiceContext, ProviderResult} from '@volar/language-service';
import {LanguageService} from '@volar/language-service';
import {URI} from 'vscode-uri';
import type ts from "tsc-ide-plugin/tsserverlibrary.shim";
import {CancellationToken} from "vscode-jsonrpc/lib/common/cancellation"

export async function handleCustomTsServerCommand(ts: typeof import("tsc-ide-plugin/tsserverlibrary.shim"),
                                                  getService: (uri: string) => Promise<ProviderResult<LanguageService>>,
                                                  cancellationToken: CancellationToken,
                                                  commandName: string,
                                                  requestArguments: TypeScriptCustomCommandArguments): Promise<ts.server.HandlerResponse | undefined> {
  const tsCancellationToken = {
    isCancellationRequested() {
      // TODO - this cancellation token apparently doesn't work,
      //        most likely because the cancel notification needs to be
      //        processed, but TypeScript language service is non-async,
      //        and does not give a chance to do any processing.
      return cancellationToken.isCancellationRequested
    },
    throwIfCancellationRequested() {
      if (this.isCancellationRequested()) {
        throw new ts.OperationCanceledException()
      }
    }
  }
  return tryHandleCustomTsServerCommandLsp(ts, commandName, requestArguments, {
    cancellationToken: tsCancellationToken,
    async process<T>(fileUri: string, range: Range | undefined, processor: (context: LspProcessingContext) => T): Promise<T | undefined> {
      const languageService = await getService(fileUri);
      if (!languageService) return undefined;
      const context: LanguageServiceContext = languageService.context;
      return await callInMapper(context, tsCancellationToken, fileUri, range, (requestContext: LspProcessingContext) => {
        return processor(requestContext)
      })
    }
  })
}

async function callInMapper<T>(
  context: LanguageServiceContext,
  cancellationToken: ts.CancellationToken,
  rawUri: string,
  range: Range | undefined,
  callable: (requestContext: LspProcessingContext) => T,
): Promise<T | undefined> {

  if (context.project.typescript == null) return undefined;

  const uri = URI.parse(rawUri);

  let languageService = getLanguageService(context) as (ts.LanguageService | undefined)
  if (!languageService) return undefined;

  let program = languageService.getProgram();
  if (!program) return undefined;

  const fileName = asFileName(context.project.typescript, uri)
  const sourceFile = program.getSourceFile(fileName)
  if (!sourceFile) return undefined;

  let docs = getDocs(context, uri);
  if (docs && range) {
    range = getGeneratedRange(docs, range);
  }

  const reverseMapper: ReverseMapper = (targetFile: ts.SourceFile, generatedRange: Range) => {
    const targetName = targetFile.fileName;
    if (sourceFile == targetFile) {
      return doReverseMap(docs, generatedRange, targetName);
    }
    else {
      const targetUri = asUri(context.project.typescript!, targetName);
      const docs = getDocs(context, targetUri);
      return doReverseMap(docs, generatedRange, targetName);
    }
  }

  return callable({languageService, program, sourceFile, range, reverseMapper, cancellationToken});
}

function doReverseMap(docs: DocumentsAndMap | undefined, generatedRange: Range, targetName: string) {
  if (!docs) return undefined;

  const sourceRange = getSourceRange(docs, generatedRange);
  if (sourceRange) {
    return {
      sourceRange,
      fileName: targetName
    }
  }
}